<?php
// core/page_renderer.php
require_once __DIR__ . '/db.php';
require_once __DIR__ . '/helpers.php';
require_once __DIR__ . '/options.php';

/* ============================================================
   SAFE SCHEMA HELPERS
   ============================================================ */
function _tbl_exists($t){
  return (int)DB::run(
    "SELECT COUNT(*) FROM information_schema.tables
     WHERE table_schema = DATABASE() AND table_name = ?", [$t]
  )->fetchColumn() > 0;
}
function _col_exists($t,$c){
  return (int)DB::run(
    "SELECT COUNT(*) FROM information_schema.columns
     WHERE table_schema = DATABASE() AND table_name=? AND column_name=?", [$t,$c]
  )->fetchColumn() > 0;
}

/* ============================================================
   HEADER MENU ITEMS (site_menus -> categories fallback)
   ============================================================ */
function _header_menu_items(): array {
  $items = [];

  // Prefer site_menus if present and has usable columns
  if (_tbl_exists('site_menus')) {
    $labelCandidates = ['label','name','title','text'];
    $urlCandidates   = ['url','link','href','path'];
    $labelCol = null; foreach ($labelCandidates as $c) if (_col_exists('site_menus',$c)) { $labelCol=$c; break; }
    $urlCol   = null; foreach ($urlCandidates as $c) if (_col_exists('site_menus',$c)) { $urlCol=$c; break; }

    if ($labelCol && $urlCol) {
      $sql = "SELECT id,$labelCol AS name,$urlCol AS url FROM site_menus";
      $w = [];
      if (_col_exists('site_menus','active'))    $w[] = "active=1";
      if (_col_exists('site_menus','is_active')) $w[] = "is_active=1";
      if ($w) $sql .= " WHERE ".implode(' AND ',$w);
      $sql .= " ORDER BY ".(_col_exists('site_menus','sort_order') ? 'sort_order' : 'id');

      foreach (DB::run($sql)->fetchAll() as $r) {
        if (!empty($r['name']) && !empty($r['url'])) {
          $items[] = ['name'=>$r['name'], 'url'=>$r['url']];
        }
      }
    }
  }

  // Fallback to categories
  if (!$items && _tbl_exists('categories')) {
    $nameCol = _col_exists('categories','name_bn') ? 'name_bn' : (_col_exists('categories','name') ? 'name' : null);
    $slugCol = _col_exists('categories','slug') ? 'slug' : null;
    if ($nameCol && $slugCol) {
      $sql = "SELECT id,$nameCol AS name,$slugCol AS slug FROM categories";
      $w = [];
      if (_col_exists('categories','active'))        $w[] = "active=1";
      elseif (_col_exists('categories','is_active')) $w[] = "is_active=1";
      elseif (_col_exists('categories','status'))    $w[] = "status IN ('1','Yes','yes','active','published')";
      if ($w) $sql .= " WHERE ".implode(' AND ',$w);
      $sql .= " ORDER BY ".(_col_exists('categories','sort_order') ? 'sort_order' : 'id');

      foreach (DB::run($sql)->fetchAll() as $r) {
        $items[] = ['name'=>$r['name'], 'url'=>"/public/category.php?slug=".rawurlencode($r['slug'])];
      }
    }
  }

  return $items;
}

/* ============================================================
   HEADER renderer — reads header_layout (rows/cols/widgets)
   Fallback to header_blocks
   ============================================================ */
function appearance_render_header(){
  $layout = option_get('header_layout', []);  // rows/cols/widgets
  $blocks = option_get('header_blocks', []);  // legacy

  // Global CSS once
  echo '<style>
    :root{ --wrap:1160px }
    .container{ max-width:var(--wrap); margin:0 auto; padding:0 12px }
    header.site-header{ background:#fff; border-bottom:1px solid #eee }
    .h-row{ display:flex; gap:16px; align-items:center; padding:10px 0 }
    .h-col{ display:flex; gap:14px; align-items:center; flex-wrap:wrap }
    .h-col.spread{ justify-content:space-between }
    .h-logo img{ height:60px }
    nav.h-menu ul{ display:flex; gap:22px; list-style:none; margin:0; padding:0; flex-wrap:wrap; justify-content:center }
    nav.h-menu a{ text-decoration:none; color:#111; font-weight:600 }
    form.h-search input{ padding:6px 10px; border:1px solid #ddd; border-radius:8px }
    .h-lang a,.h-social a{ text-decoration:none; margin-left:10px }
    .va-main{ max-width:var(--wrap); margin:24px auto; padding:0 12px }
    .blk-html h1,.blk-html h2,.blk-html h3 { text-align:center }
    @media (max-width:768px){
      .h-logo img{ height:44px }
      nav.h-menu ul{ gap:14px }
    }
  </style>';

  // If a rows/columns layout exists, render it
  if (is_array($layout) && count($layout)) {
    echo '<header class="site-header"><div class="container">';
    foreach ($layout as $row) {
      echo '<div class="h-row">';
      $cols = isset($row['columns']) && is_array($row['columns']) ? $row['columns'] : [];
      foreach ($cols as $col) {
        $w = isset($col['width']) ? (int)$col['width'] : 100;
        if ($w < 5) $w = 5; if ($w > 100) $w = 100;
        $spread = !empty($col['spread']) ? ' spread' : '';
        echo '<div class="h-col'.$spread.'" style="flex:0 0 '.$w.'%">';
        $widgets = isset($col['widgets']) && is_array($col['widgets']) ? $col['widgets'] : [];
        foreach ($widgets as $wgt) {
          $type = $wgt['type'] ?? 'html';
          $cfg  = isset($wgt['cfg']) && is_array($wgt['cfg']) ? $wgt['cfg'] : [];

          if ($type === 'logo') {
            $src  = $cfg['image'] ?? ($cfg['url'] ?? '/assets/logo.png');
            $href = $cfg['link']  ?? '/';
            echo '<a class="h-logo" href="'.e($href).'"><img src="'.e($src).'" alt=""></a>';

          } elseif ($type === 'menu') {
            $items = _header_menu_items();
            echo '<nav class="h-menu"><ul>';
            foreach ($items as $it) echo '<li><a href="'.e($it['url']).'">'.e($it['name']).'</a></li>';
            echo '</ul></nav>';

          } elseif ($type === 'search') {
            $ph = $cfg['placeholder'] ?? 'খুঁজুন';
            echo '<form class="h-search" action="/public/search.php"><input name="q" placeholder="'.e($ph).'"></form>';

          } elseif ($type === 'lang') {
            $its = isset($cfg['items']) && is_array($cfg['items']) ? $cfg['items'] : [['label'=>'Eng','href'=>'/en']];
            echo '<div class="h-lang">';
            foreach ($its as $it) echo '<a href="'.e($it['href']??'#').'">'.e($it['label']??'').'</a>';
            echo '</div>';

          } elseif ($type === 'social') {
            $links = isset($cfg['links']) && is_array($cfg['links']) ? $cfg['links'] : [];
            echo '<div class="h-social">';
            foreach ($links as $a) echo '<a target="_blank" href="'.e($a['url']??'#').'">'.e($a['label']??'•').'</a>';
            echo '</div>';

          } elseif ($type === 'spacer') {
            echo '<div style="flex:1"></div>';

          } else { // html
            echo '<div class="blk-html">'.($cfg['html'] ?? '').'</div>';
          }
        }
        echo '</div>'; // .h-col
      }
      echo '</div>'; // .h-row
    }
    echo '</div></header>';
    return;
  }

  // Fallback: legacy flat header_blocks
  echo '<header class="site-header"><div class="container h-row" style="justify-content:center">';
  if (is_array($blocks)) {
    foreach ($blocks as $b) {
      $type = $b['type'] ?? ''; $cfg = isset($b['cfg']) && is_array($b['cfg']) ? $b['cfg'] : [];
      if ($type==='logo'){
        $src=$cfg['url']??'/assets/logo.png'; $href=$cfg['link']??'/';
        echo '<a class="h-logo" href="'.e($href).'"><img src="'.e($src).'" alt=""></a>';
      } elseif ($type==='menu'){
        $items=_header_menu_items(); echo '<nav class="h-menu"><ul>';
        foreach($items as $it) echo '<li><a href="'.e($it['url']).'">'.e($it['name']).'</a></li>';
        echo '</ul></nav>';
      } elseif ($type==='search'){
        echo '<form class="h-search" action="/public/search.php"><input name="q" placeholder="খুঁজুন"></form>';
      } elseif ($type==='lang'){
        echo '<div class="h-lang"><a href="/en">Eng</a></div>';
      } elseif ($type==='social'){
        echo '<div class="h-social"><a href="#">FB</a><a href="#">YT</a></div>';
      } elseif ($type==='html'){
        echo $cfg['html']??'';
      }
    }
  }
  echo '</div></header>';
}

/* alias if old code still calls render_header() */
if (!function_exists('render_header')) {
  function render_header(){ appearance_render_header(); }
}

/* ============================================================
   EXTRA HELPERS used by block rendering
   ============================================================ */
function db_has_col($table,$col){
  return (int)DB::run(
    "SELECT COUNT(*) FROM information_schema.COLUMNS
     WHERE TABLE_SCHEMA = DATABASE() AND TABLE_NAME=? AND COLUMN_NAME=?",
    [$table,$col]
  )->fetchColumn() > 0;
}
function pick_col($table, $cands){
  foreach ($cands as $c) if (db_has_col($table,$c)) return $c;
  return null;
}

/* ============================================================
   FOOTER
   ============================================================ */
function render_footer(){
  $blocks = option_get('footer_blocks', []);
  ob_start(); ?>
  <footer class="va-footer">
    <div class="container">
      <?php if (is_array($blocks)) foreach ($blocks as $b):
        $type = $b['type'] ?? 'html';
        $cfg  = isset($b['cfg']) && is_array($b['cfg']) ? $b['cfg'] : [];
        if ($type==='html'){
          echo '<div class="f-html">'.($cfg['html'] ?? ($b['html'] ?? '')).'</div>';
        } elseif ($type==='menu'){
          $items = $cfg['items'] ?? []; echo '<ul class="f-menu">';
          foreach($items as $it) echo '<li><a href="'.e($it['url']??'#').'">'.e($it['label']??'').'</a></li>';
          echo '</ul>';
        } elseif ($type==='social'){
          $links = $cfg['links'] ?? []; echo '<div class="f-social">';
          foreach($links as $a) echo '<a href="'.e($a['url']??'#').'" target="_blank">'.e($a['label']??'•').'</a>';
          echo '</div>';
        } else {
          echo '<div class="f-any">'.e($type).'</div>';
        }
      endforeach; ?>
      <div class="f-copy">© <?=date('Y')?> <?=e(option_get('site_name','ভোরের আকাশ'))?></div>
    </div>
  </footer>
  <style>
    .va-footer{background:#e9e9e9;margin-top:30px}
    .va-footer .container{max-width:1160px;margin:0 auto;padding:18px 12px}
    .f-menu{display:flex;gap:18px;list-style:none;padding:0;margin:0 0 8px}
    .f-menu a{text-decoration:none;color:#111}
    .f-copy{color:#444}
  </style>
  <?php return ob_get_clean();
}

/* ============================================================
   PAGE RENDERERS
   ============================================================ */
function render_page_by_id(int $id, bool $allowDraftPreview=false){
  $row = DB::run("SELECT * FROM pages WHERE id=?",[$id])->fetch();
  if (!$row) return '';
  if (($row['status']??'draft')!=='published' && !$allowDraftPreview) return '';
  return _render_page_row($row);
}
function render_page_by_slug(string $slug, bool $allowDraftPreview=false){
  $row = DB::run("SELECT * FROM pages WHERE slug=?",[$slug])->fetch();
  if (!$row) return '';
  if (($row['status']??'draft')!=='published' && !$allowDraftPreview) return '';
  return _render_page_row($row);
}

function _render_page_row(array $row){
  $blocks = json_decode($row['blocks'] ?? '[]', true) ?: [];
  $html   = [];

  if (!empty($row['show_header'])) $html[] = render_header();

  $title = $row['meta_title'] ?: ($row['title'] ?? '');
  $desc  = $row['meta_desc']  ?: '';
  $html[] = "<title>".e($title)."</title>";
  if ($desc) $html[] = '<meta name="description" content="'.e($desc).'">';

  // centered, responsive main container
  $html[] = '<main class="va-main container">';
  if (!$blocks) $html[] = '<p>No content yet.</p>';
  foreach ($blocks as $b){
    $type = $b['type'] ?? 'html';
    $cfg  = isset($b['cfg']) && is_array($b['cfg']) ? $b['cfg'] : $b;

    if ($type==='html'){
      $html[] = '<section class="blk-html">'.($cfg['html'] ?? '').'</section>';
    }
    elseif ($type==='shortcode'){
      $html[] = '<section class="blk-shortcode">'.e($cfg['code'] ?? '').'</section>';
    }
    elseif ($type==='category'){
      $html[] = _render_category_section($cfg);
    }
    elseif ($type==='posts'){
      $html[] = _render_category_section([
        'category_id'=>null,
        'cat_slug'=>$cfg['cat_slug']??null,
        'limit'=>$cfg['limit']??8,
        'layout'=>$cfg['layout']??'grid',
        'title'=>$cfg['title']??''
      ]);
    }
    else {
      $html[] = '<section>'.e($type).'</section>';
    }
  }
  $html[] = '</main>';

  if (!empty($row['show_footer'])) $html[] = render_footer();

  return implode("\n", $html);
}

/* ============================================================
   BLOCK: CATEGORY / POSTS
   ============================================================ */
function _render_category_section(array $cfg){
  $limit  = max(1, (int)($cfg['limit'] ?? 6));
  $layout = $cfg['layout'] ?? 'grid';
  $title  = trim($cfg['title'] ?? '');

  $titleCol = pick_col('posts', ['title_bn','title']);
  $imgCol   = pick_col('posts', ['featured_image','image','photo']);
  $slugCol  = pick_col('posts', ['slug']);
  $catCol   = pick_col('posts', ['category_id']);
  if (!$titleCol) return '';

  $where = '1=1';
  $bind  = [];
  if (!empty($cfg['category_id']) && $catCol){
    $where = "$catCol=?";
    $bind[] = (int)$cfg['category_id'];
  } elseif (!empty($cfg['cat_slug'])) {
    $catSlugCol = db_has_col('categories','slug') ? 'slug' : null;
    if ($catSlugCol){
      $cid = DB::run("SELECT id FROM categories WHERE $catSlugCol=? LIMIT 1",[$cfg['cat_slug']])->fetchColumn();
      if ($cid) { $where = ($catCol ? "$catCol=?" : '1=1'); $bind[] = (int)$cid; }
    }
  }

  $rows = DB::run(
    "SELECT id,$titleCol".($imgCol?",$imgCol":"").($slugCol?",$slugCol":"")."
     FROM posts WHERE $where ORDER BY id DESC LIMIT $limit",
    $bind
  )->fetchAll();

  ob_start(); ?>
  <section class="blk-posts" style="margin:18px 0">
    <?php if ($title): ?><h3 class="blk-title" style="margin:0 0 10px;border-left:4px solid #f90;padding-left:8px"><?=e($title)?></h3><?php endif; ?>
    <div class="<?= $layout==='list' ? 'list' : 'grid' ?>" style="<?= $layout==='list'
      ? 'display:flex;flex-direction:column;gap:10px'
      : 'display:grid;grid-template-columns:repeat(3,1fr);gap:14px' ?>">
      <?php foreach ($rows as $r): ?>
        <article class="card" style="border:1px solid #eee;border-radius:12px;overflow:hidden;background:#fff">
          <?php if (!empty($imgCol) && !empty($r[$imgCol])): ?>
            <div class="thumb" style="height:160px;background:#f5f5f5;display:flex;align-items:center;justify-content:center">
              <img src="<?=e($r[$imgCol])?>" alt="" style="max-width:100%;max-height:100%">
            </div>
          <?php endif; ?>
          <h4 class="t" style="font-size:15px;line-height:1.35;margin:8px 10px 12px"><?=e($r[$titleCol])?></h4>
        </article>
      <?php endforeach; ?>
    </div>
  </section>
  <?php
  return ob_get_clean();
}
