<?php
// core/auth.php
require_once __DIR__ . '/db.php';
require_once __DIR__ . '/helpers.php';

if (session_status() !== PHP_SESSION_ACTIVE) {
  session_start();
}

/**
 * Return the currently authenticated user (array) or null.
 */
function current_user() {
  return $_SESSION['user'] ?? null;
}

/**
 * Compute the admin login URL relative to the current script.
 * Avoids base_url() (which points to /public).
 */
function admin_login_url(): string {
  // e.g. /.../admin/dashboard.php → /.../admin/index.php
  $script = $_SERVER['SCRIPT_NAME'] ?? '/admin/index.php';
  $dir    = rtrim(str_replace('\\', '/', dirname($script)), '/');
  // If we're not already inside /admin, force /admin/index.php
  if (substr($dir, -6) !== '/admin') {
    $dir .= '/admin';
  }
  return $dir . '/index.php';
}

/**
 * Require a logged-in user for admin pages.
 * If not logged in, redirect to /admin/index.php (same directory).
 */
function require_login() {
  if (!current_user()) {
    header('Location: ' . admin_login_url());
    exit;
  }
}

/**
 * Authenticate and establish a session.
 * - Trims email to avoid invisible whitespace issues.
 * - Regenerates session ID on success (prevents fixation).
 */
function login($email, $pass): bool {
  $email = trim((string)$email);
  $pass  = (string)$pass;

  $u = DB::run('SELECT * FROM users WHERE email = ? LIMIT 1', [$email])->fetch();
  if ($u && password_verify($pass, $u['password'])) {
    if (session_status() === PHP_SESSION_ACTIVE) {
      session_regenerate_id(true);
    }
    $_SESSION['user'] = [
      'id'    => $u['id'],
      'name'  => $u['name'],
      'role'  => $u['role'],
      'email' => $u['email']
    ];
    return true;
  }
  return false;
}

/**
 * Log out the user and destroy the session.
 */
function logout() {
  $_SESSION = [];
  if (ini_get('session.use_cookies')) {
    $p = session_get_cookie_params();
    setcookie(session_name(), '', time() - 42000, $p['path'], $p['domain'], $p['secure'], $p['httponly']);
  }
  session_destroy();
}

/**
 * Require that the current user has one of the allowed roles.
 */
function require_role($role) {
  $u = current_user();
  if (!$u || !in_array($u['role'], (array)$role, true)) {
    http_response_code(403);
    die('Forbidden');
  }
}
