<?php
// admin/video.php — list + quick add/edit for videos

require_once __DIR__ . '/../core/auth.php';
require_login();
require_role(['admin']);

require_once __DIR__ . '/../core/db.php';
require_once __DIR__ . '/../core/helpers.php';

$SELF = basename(__FILE__);

/* ---------------- helpers ---------------- */
function table_exists($t){return (int)DB::run("SELECT COUNT(*) FROM information_schema.TABLES WHERE TABLE_SCHEMA=DATABASE() AND TABLE_NAME=?",[$t])->fetchColumn()>0;}
function col_exists($t,$c){return (int)DB::run("SELECT COUNT(*) FROM information_schema.COLUMNS WHERE TABLE_SCHEMA=DATABASE() AND TABLE_NAME=? AND COLUMN_NAME=?",[$t,$c])->fetchColumn()>0;}
function human_time($ts){
  $s = time() - strtotime($ts);
  if ($s < 60) return "just now";
  $m = floor($s/60); if ($m < 60) return $m." min ago";
  $h = floor($m/60); if ($h < 24) return $h." hours ago";
  $d = floor($h/24); if ($d < 30) return $d." days ago";
  $mo = floor($d/30); if ($mo < 12) return $mo." months ago";
  $y = floor($mo/12); return $y." years ago";
}
function extract_youtube_id($input){
  if (preg_match('~(?:youtu\.be/|youtube\.com/(?:watch\?v=|embed/|shorts/))([\w\-]{6,})~i',$input,$m)) return $m[1];
  return preg_replace('/[^A-Za-z0-9_\-]/','', $input);
}

/* ---------------- ensure tables ---------------- */
if (!table_exists('video_categories')) {
  DB::run("CREATE TABLE video_categories (
    id INT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    name VARCHAR(150) NOT NULL,
    created_at TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP
  ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4");
}
if (!table_exists('videos')) {
  DB::run("CREATE TABLE videos (
    id INT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    category_id INT UNSIGNED NULL,
    video_key VARCHAR(64) NOT NULL,      -- YouTube ID or similar
    title VARCHAR(255) NULL,
    is_active TINYINT(1) NOT NULL DEFAULT 1,
    created_at TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP NULL DEFAULT NULL ON UPDATE CURRENT_TIMESTAMP,
    KEY idx_cat (category_id),
    KEY idx_active (is_active),
    CONSTRAINT fk_videos_cat FOREIGN KEY (category_id)
      REFERENCES video_categories(id) ON DELETE SET NULL ON UPDATE CASCADE
  ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4");
} else {
  if (!col_exists('videos','video_key')) DB::run("ALTER TABLE videos ADD COLUMN video_key VARCHAR(64) NOT NULL AFTER category_id");
  if (!col_exists('videos','title')) DB::run("ALTER TABLE videos ADD COLUMN title VARCHAR(255) NULL AFTER video_key");
  if (!col_exists('videos','is_active')) DB::run("ALTER TABLE videos ADD COLUMN is_active TINYINT(1) NOT NULL DEFAULT 1");
  if (!col_exists('videos','created_at')) DB::run("ALTER TABLE videos ADD COLUMN created_at TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP");
  if (!col_exists('videos','updated_at')) DB::run("ALTER TABLE videos ADD COLUMN updated_at TIMESTAMP NULL DEFAULT NULL ON UPDATE CURRENT_TIMESTAMP");
}

/* ---------------- actions (toggle/delete/save) ---------------- */
if (isset($_GET['toggle'])) {
  DB::run("UPDATE videos SET is_active=1-is_active WHERE id=?", [(int)$_GET['toggle']]);
  header("Location: {$SELF}"); exit;
}
if (isset($_GET['del'])) {
  DB::run("DELETE FROM videos WHERE id=?", [(int)$_GET['del']]);
  header("Location: {$SELF}"); exit;
}

$cats = DB::run("SELECT id,name FROM video_categories ORDER BY name ASC")->fetchAll(PDO::FETCH_KEY_PAIR);

$errors = [];
$editing = null;
if (isset($_GET['edit'])) {
  $editing = (int)$_GET['edit'];
  $row = DB::run("SELECT * FROM videos WHERE id=?", [$editing])->fetch();
  if ($row) {
    $_POST['video_key'] = $row['video_key'];
    $_POST['title']     = $row['title'];
    $_POST['category_id']= $row['category_id'];
  }
}

if ($_SERVER['REQUEST_METHOD']==='POST') {
  $video_key  = extract_youtube_id(trim($_POST['video_key'] ?? ''));
  $title      = trim($_POST['title'] ?? '');
  $category_id= ($_POST['category_id'] ?? '') !== '' ? (int)$_POST['category_id'] : null;

  if ($video_key === '') $errors[]='Please provide a YouTube URL or ID.';
  if (!$errors) {
    if (isset($_POST['id']) && $_POST['id']!=='') {
      DB::run("UPDATE videos SET video_key=?, title=?, category_id=? WHERE id=?",
        [$video_key, $title ?: null, $category_id, (int)$_POST['id']]);
    } else {
      DB::run("INSERT INTO videos (video_key,title,category_id) VALUES (?,?,?)",
        [$video_key, $title ?: null, $category_id]);
    }
    header("Location: {$SELF}?saved=1"); exit;
  }
}

/* ---------------- data ---------------- */
$rows = DB::run("
  SELECT v.*, vc.name AS cat_name
  FROM videos v
  LEFT JOIN video_categories vc ON vc.id = v.category_id
  ORDER BY v.id DESC
")->fetchAll(PDO::FETCH_ASSOC);

/* ---------------- UI ---------------- */
include __DIR__ . '/partials/header.php';
include __DIR__ . '/partials/sidebar.php';
?>
<style>
.badge{display:inline-block;padding:4px 10px;border-radius:999px;font-size:12px}
.badge.on{background:#10b981;color:#fff}
.badge.off{background:#ef4444;color:#fff}
.btn{display:inline-block;background:#0f172a;color:#fff;border:none;border-radius:10px;padding:10px 14px;text-decoration:none}
.btn-chip{display:inline-block;padding:6px 10px;border-radius:10px;background:#0f172a;color:#fff;text-decoration:none}
.btn-chip.secondary{background:#111827}
.panel{background:#fff;border:1px solid #e5e7eb;border-radius:12px;padding:18px}
.table .link{color:#2563eb;text-decoration:underline}
.form-inline{background:#fff;border:1px solid #e5e7eb;border-radius:12px;padding:16px;margin-bottom:14px}
.input, select{width:100%;padding:10px;border:1px solid #d1d5db;border-radius:8px;background:#fff}
.grid{display:grid;grid-template-columns:260px 1fr 220px 120px;gap:10px}
</style>

<div class="admin-container">
  <div style="display:flex;align-items:center;justify-content:space-between;gap:12px;margin-bottom:10px">
    <h2>All video</h2>
    <a class="btn" href="<?= e($SELF) ?>?new=1">+ New video</a>
  </div>

  <?php if ($errors): ?>
    <div class="panel" style="border-color:#fecaca;background:#fef2f2;color:#7f1d1d"> <?= e(implode(' ', $errors)) ?> </div>
  <?php elseif (isset($_GET['saved'])): ?>
    <div class="panel" style="border-color:#bbf7d0;background:#f0fdf4;color:#166534"> Saved. </div>
  <?php endif; ?>

  <?php if (isset($_GET['new']) || $editing): ?>
    <form class="form-inline" method="post">
      <input type="hidden" name="id" value="<?= $editing ? (int)$editing : '' ?>">
      <div class="grid">
        <div>
          <label style="display:block;margin-bottom:6px">Category</label>
          <select name="category_id">
            <option value="">— None —</option>
            <?php foreach ($cats as $cid => $cname): ?>
              <option value="<?= (int)$cid ?>" <?= (isset($_POST['category_id']) && (int)$_POST['category_id']===(int)$cid)?'selected':''; ?>>
                <?= e($cname) ?>
              </option>
            <?php endforeach; ?>
          </select>
        </div>
        <div>
          <label style="display:block;margin-bottom:6px">Video link / ID</label>
          <input class="input" type="text" name="video_key" placeholder="https://youtu.be/xxxxx or xxxxx"
                 value="<?= e($_POST['video_key'] ?? '') ?>" required>
        </div>
        <div>
          <label style="display:block;margin-bottom:6px">Details (optional)</label>
          <input class="input" type="text" name="title" placeholder="Short description"
                 value="<?= e($_POST['title'] ?? '') ?>">
        </div>
        <div style="align-self:end">
          <button class="btn" type="submit"><?= $editing ? 'Update' : 'Save' ?></button>
        </div>
      </div>
    </form>
  <?php endif; ?>

  <div class="panel">
    <table class="table">
      <thead>
        <tr>
          <th>SL</th>
          <th>Link</th>
          <th>Details</th>
          <th>Uploaded at</th>
          <th>Manage</th>
        </tr>
      </thead>
      <tbody>
      <?php if ($rows): $i=1; foreach ($rows as $r): ?>
        <tr>
          <td><?= $i++ ?></td>
          <td>
            <?php $url = 'https://youtu.be/'.e($r['video_key']); ?>
            <a class="link" href="<?= $url ?>" target="_blank"><?= e($r['video_key']) ?></a>
            <?php if (!empty($r['cat_name'])): ?>
              <div style="color:#64748b;font-size:12px;margin-top:2px"><?= e($r['cat_name']) ?></div>
            <?php endif; ?>
          </td>
          <td><?= e($r['title'] ?: '…') ?></td>
          <td><?= e(human_time($r['created_at'])) ?></td>
          <td>
            <a class="btn-chip" href="<?= e($SELF) ?>?toggle=<?= (int)$r['id'] ?>">
              <span><?= $r['is_active'] ? 'Turn Off' : 'Turn On' ?></span>
            </a>
            <a class="btn-chip" href="<?= e($SELF) ?>?edit=<?= (int)$r['id'] ?>">Edit</a>
            <a class="btn-chip secondary" href="javascript:void(0)"
               onclick="if(confirm('Delete this video?')) location.href='<?= e($SELF) ?>?del=<?= (int)$r['id'] ?>'">
               Delete
            </a>
          </td>
        </tr>
      <?php endforeach; else: ?>
        <tr><td colspan="5">No videos yet.</td></tr>
      <?php endif; ?>
      </tbody>
    </table>
  </div>
</div>

<?php include __DIR__ . '/partials/footer.php';
