<?php
// admin/trending_tags.php
require_once __DIR__ . '/../core/auth.php';
require_login();
require_role(['admin']);

require_once __DIR__ . '/../core/db.php';
require_once __DIR__ . '/../core/helpers.php';

// ---- fetch all tags
$allTags = DB::run("SELECT id, name FROM tags ORDER BY name ASC")->fetchAll(PDO::FETCH_ASSOC);

// ---- currently active trending tags (to preselect)
$active = DB::run("SELECT tag_id FROM trending_tags WHERE is_active=1 ORDER BY position ASC")
            ->fetchAll(PDO::FETCH_COLUMN);
$active = array_map('intval', $active);

// ---- handle save
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $ids = isset($_POST['tag_ids']) ? (array)$_POST['tag_ids'] : [];
    // normalize to ints and unique
    $ids = array_values(array_unique(array_map(fn($v)=> (int)$v, $ids)));

    DB::beginTransaction();
    try {
        // deactivate everything first so "removals" are reflected
        DB::run("UPDATE trending_tags SET is_active=0");

        // upsert each selected tag with its new position
        $pos = 1;
        $stmt = DB::prepare(
            "INSERT INTO trending_tags (tag_id, position, is_active)
             VALUES (?, ?, 1)
             ON DUPLICATE KEY UPDATE position=VALUES(position), is_active=1"
        );
        foreach ($ids as $tid) {
            $stmt->execute([$tid, $pos++]);
        }

        DB::commit();
        $_SESSION['flash'] = ['type'=>'success','msg'=>'Trending tags updated.'];
        header('Location: trending_tags.php');
        exit;
    } catch (Throwable $e) {
        DB::rollBack();
        $_SESSION['flash'] = ['type'=>'error','msg'=>'Save failed: '.$e->getMessage()];
    }
}

// ---- view helpers
function is_selected(int $id, array $selected): string {
    return in_array($id, $selected, true) ? 'selected' : '';
}

include __DIR__ . '/partials/header.php';
include __DIR__ . '/partials/sidebar.php';  // opens wrapper
?>
<style>
.tag-layout{display:grid;grid-template-columns: 1fr 1fr; gap:18px}
.tag-card{background:#fff;border:1px solid #e5e7eb;border-radius:10px;padding:16px}
.tag-card h3{margin:0 0 10px 0;font-size:18px}
.select-tags{width:100%;min-height:480px}
.flash{padding:10px;border-radius:8px;margin-bottom:12px}
.flash.success{background:#ecfdf5;color:#065f46;border:1px solid #a7f3d0}
.flash.error{background:#fef2f2;color:#991b1b;border:1px solid #fecaca}
.help{font-size:12px;color:#6b7280;margin-top:6px}
.table{width:100%;border-collapse:collapse}
.table th,.table td{border-bottom:1px solid #eee;padding:8px 6px;text-align:left}
.btn{display:inline-block;background:#111827;color:#fff;border-radius:8px;padding:10px 14px;text-decoration:none}
.btn:hover{opacity:.9}
</style>

<div class="admin-container">
  <h2>Trending Tag</h2>

  <?php if (!empty($_SESSION['flash'])): $f=$_SESSION['flash']; unset($_SESSION['flash']); ?>
    <div class="flash <?= e($f['type']) ?>"><?= e($f['msg']) ?></div>
  <?php endif; ?>

  <div class="tag-layout">
    <div class="tag-card">
      <h3>Select</h3>
      <form method="post">
        <!-- Simple multi-select. If you want token UI later, drop in TomSelect/Choices.js -->
        <select class="select-tags" name="tag_ids[]" multiple size="22" aria-label="Trending tags">
          <?php foreach($allTags as $t): ?>
            <option value="<?= (int)$t['id'] ?>" <?= is_selected((int)$t['id'], $active) ?>>
              <?= e($t['name']) ?>
            </option>
          <?php endforeach; ?>
        </select>
        <div class="help">Hold Ctrl/⌘ to select multiple. The order you submit is saved as rank.</div>
        <div style="margin-top:10px">
          <button class="btn" type="submit">submit</button>
        </div>
      </form>
    </div>

    <div class="tag-card">
      <h3>All Tags</h3>
      <table class="table">
        <thead><tr><th style="width:70px">SL</th><th>Name</th></tr></thead>
        <tbody>
          <?php $i=1; foreach($allTags as $t): ?>
            <tr>
              <td><?= $i++ ?></td>
              <td><?= e($t['name']) ?></td>
            </tr>
          <?php endforeach; if (!$allTags): ?>
            <tr><td colspan="2">No tags yet.</td></tr>
          <?php endif; ?>
        </tbody>
      </table>
    </div>
  </div>
</div>

<?php include __DIR__ . '/partials/footer.php'; // closes wrapper ?>
