<?php
// admin/trending_tag.php
// Manage the list and order of “Trending” tags

require_once __DIR__ . '/../core/auth.php';
require_login();
require_role(['admin']);

require_once __DIR__ . '/../core/db.php';
require_once __DIR__ . '/../core/helpers.php';

/* ---------- helpers ---------- */
function table_exists(string $t): bool {
  return (int) DB::run(
    "SELECT COUNT(*) FROM information_schema.TABLES
     WHERE TABLE_SCHEMA = DATABASE() AND TABLE_NAME=?",
    [$t]
  )->fetchColumn() > 0;
}

/* ---------- ensure table ---------- */
if (!table_exists('trending_tags')) {
  DB::run("
    CREATE TABLE IF NOT EXISTS trending_tags (
      id INT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
      tag_id INT UNSIGNED NOT NULL,
      position INT UNSIGNED NOT NULL DEFAULT 0,
      created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
      UNIQUE KEY uniq_trending_tag (tag_id),
      CONSTRAINT fk_trending_tag FOREIGN KEY (tag_id) REFERENCES tags(id) ON DELETE CASCADE
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4
  ");
}

/* ---------- read data ---------- */
$allTags = DB::run("SELECT id, name FROM tags ORDER BY name ASC")->fetchAll();
$selectedIds = DB::run("SELECT tag_id FROM trending_tags ORDER BY position ASC")
  ->fetchAll(PDO::FETCH_COLUMN);

/* ---------- save ---------- */
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
  $csv = trim($_POST['tag_ids'] ?? '');
  $ids = array_values(array_filter(array_map('intval', array_unique(array_map('trim', explode(',', $csv))))));

  // replace all rows to keep order simple
  DB::run("DELETE FROM trending_tags");
  $pos = 1;
  foreach ($ids as $tid) {
    DB::run("INSERT INTO trending_tags (tag_id, position) VALUES (?,?)", [$tid, $pos++]);
  }
  header("Location: trending_tag.php?saved=1");
  exit;
}

include __DIR__ . '/partials/header.php';
include __DIR__ . '/partials/sidebar.php';
?>
<style>
/* small, self-contained styles for chips + controls */
.chips {display:flex;flex-wrap:wrap;gap:8px;min-height:42px;padding:8px;border:1px solid #e5e7eb;border-radius:8px;background:#fff}
.chip{background:#2563eb;color:#fff;border-radius:999px;padding:6px 10px;display:inline-flex;align-items:center;gap:8px;font-size:13px}
.chip button{border:0;background:transparent;color:#fff;font-weight:700;cursor:pointer;line-height:1}
.chip.active{outline:2px solid #60a5fa}
.controls{display:flex;gap:8px;margin:8px 0}
.controls .btn-sm{padding:6px 10px;border-radius:8px;border:1px solid #cbd5e1;background:#f8fafc;cursor:pointer}
.controls .btn-sm:hover{background:#eef2f7}
select.mult{width:100%;min-height:260px;padding:8px;border-radius:8px;border:1px solid #e5e7eb;background:#fff}
</style>

<div class="admin-container">
  <h2>Trending Tag</h2>
  <?php if (isset($_GET['saved'])): ?>
    <div class="alert success">Saved.</div>
  <?php endif; ?>

  <div class="grid-2">
    <!-- LEFT: form -->
    <div class="panel">
      <h3>Tag Form</h3>

      <form method="post" id="frmTrending">
        <label>Select</label>
        <!-- chips display (ordered) -->
        <div id="chips" class="chips" aria-live="polite"></div>

        <div class="controls">
          <button type="button" class="btn-sm" id="upBtn">↑ Move up</button>
          <button type="button" class="btn-sm" id="downBtn">↓ Move down</button>
          <button type="button" class="btn-sm" id="clearBtn">Clear</button>
        </div>

        <!-- Multi select: all tags (you can Ctrl/Cmd click to select) -->
        <select id="tagSelect" class="mult" multiple>
          <?php
            // Keep currently selected first (in stored order), then the rest alphabetically
            $printed = [];
            foreach ($selectedIds as $sid) {
              foreach ($allTags as $t) {
                if ($t['id'] == $sid) {
                  echo '<option value="'.(int)$t['id'].'" selected>'.e($t['name']).'</option>';
                  $printed[$t['id']] = true;
                  break;
                }
              }
            }
            foreach ($allTags as $t) {
              if (!isset($printed[$t['id']])) {
                echo '<option value="'.(int)$t['id'].'">'.e($t['name']).'</option>';
              }
            }
          ?>
        </select>

        <input type="hidden" name="tag_ids" id="tag_ids" value="">
        <button class="btn" style="margin-top:10px">submit</button>
      </form>
    </div>

    <!-- RIGHT: list of all tags -->
    <div class="panel">
      <h3>All Tags</h3>
      <table class="table">
        <thead><tr><th>SL</th><th>Name</th></tr></thead>
        <tbody>
        <?php $i=1; foreach ($allTags as $t): ?>
          <tr><td><?= $i++ ?></td><td><?= e($t['name']) ?></td></tr>
        <?php endforeach; if (!$allTags): ?>
          <tr><td colspan="2">No tags yet.</td></tr>
        <?php endif; ?>
        </tbody>
      </table>
    </div>
  </div>
</div>

<script>
// --- tiny “chips + order” manager (no external libs) ---
const selectEl = document.getElementById('tagSelect');
const chipsEl  = document.getElementById('chips');
const hidden   = document.getElementById('tag_ids');

let order = [];     // holds selected tag IDs in order
let active = null;  // which chip is active for up/down

function rebuildFromSelect() {
  // start from the current DB order first (server preselected), or clicked order afterwards
  order = Array.from(selectEl.selectedOptions).map(o => parseInt(o.value));
  render();
}
function render() {
  chipsEl.innerHTML = '';
  order.forEach(id => {
    const opt = Array.from(selectEl.options).find(o => parseInt(o.value) === id);
    if (!opt) return;
    const chip = document.createElement('span');
    chip.className = 'chip' + (active === id ? ' active' : '');
    chip.textContent = opt.text;
    const btn = document.createElement('button');
    btn.type = 'button';
    btn.textContent = '×';
    btn.title = 'Remove';
    btn.onclick = () => removeId(id);
    chip.onclick = () => { active = (active === id ? null : id); render(); };
    chip.appendChild(btn);
    chipsEl.appendChild(chip);
  });
  // keep <select> selection in sync
  Array.from(selectEl.options).forEach(o => o.selected = order.includes(parseInt(o.value)));
  // write CSV for submit
  hidden.value = order.join(',');
}

function removeId(id){
  order = order.filter(x => x !== id);
  if (active === id) active = null;
  render();
}

selectEl.addEventListener('change', () => {
  // add any newly selected ids to order (append at end)
  const selected = Array.from(selectEl.selectedOptions).map(o => parseInt(o.value));
  selected.forEach(id => { if (!order.includes(id)) order.push(id); });
  // also drop any unselected from order
  order = order.filter(id => selected.includes(id));
  render();
});

document.getElementById('upBtn').onclick = () => {
  if (active == null) return;
  const idx = order.indexOf(active);
  if (idx > 0) {
    [order[idx-1], order[idx]] = [order[idx], order[idx-1]];
    render();
  }
};
document.getElementById('downBtn').onclick = () => {
  if (active == null) return;
  const idx = order.indexOf(active);
  if (idx >= 0 && idx < order.length-1) {
    [order[idx+1], order[idx]] = [order[idx], order[idx+1]];
    render();
  }
};
document.getElementById('clearBtn').onclick = () => { order = []; active=null; render(); };

rebuildFromSelect(); // initial
</script>

<?php include __DIR__ . '/partials/footer.php';
