<?php
// admin/tag.php
require_once __DIR__ . '/../core/auth.php';  require_login();  require_role(['admin']);
require_once __DIR__ . '/../core/db.php';
require_once __DIR__ . '/../core/helpers.php';

/* ---------- helpers (schema-safe) ---------- */
if (!function_exists('db_table_exists')) {
  function db_table_exists(string $table): bool {
    return (int) DB::run(
      "SELECT COUNT(*) FROM information_schema.TABLES
       WHERE TABLE_SCHEMA = DATABASE() AND TABLE_NAME=?", [$table]
    )->fetchColumn() > 0;
  }
}
if (!function_exists('slugify_tag')) {
  function slugify_tag(string $s): string {
    $s = trim(mb_strtolower($s, 'UTF-8'));
    $s = preg_replace('/[^\p{L}\p{Nd}]+/u', '-', $s);
    $s = trim($s, '-');
    return $s ?: '';
  }
}

/* ---------- state ---------- */
$id  = isset($_GET['id']) ? (int)$_GET['id'] : 0;
$tag = ['name'=>'', 'slug'=>''];

if ($id) {
  $row = DB::run("SELECT id,name,slug FROM tags WHERE id=?", [$id])->fetch();
  if ($row) $tag = $row;
}

/* ---------- create / update ---------- */
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
  $name = trim($_POST['name'] ?? '');
  $slug = trim($_POST['slug'] ?? '');
  if ($slug === '') $slug = slugify_tag($name);

  if ($id) {
    DB::run("UPDATE tags SET name=?, slug=? WHERE id=?", [$name, $slug, $id]);
    header('Location: tag.php?saved=1'); exit;
  } else {
    DB::run("INSERT INTO tags (name, slug, created_at) VALUES (?, ?, NOW())", [$name, $slug]);
    header('Location: tag.php?saved=1'); exit;
  }
}

/* ---------- delete ---------- */
if (isset($_GET['del'])) {
  $del = (int) $_GET['del'];
  // If the post_tags table exists, clean mappings first
  if (db_table_exists('post_tags')) {
    DB::run("DELETE FROM post_tags WHERE tag_id=?", [$del]);
  }
  DB::run("DELETE FROM tags WHERE id=?", [$del]);
  header('Location: tag.php'); exit;
}

/* ---------- list (with safe post count) ---------- */
$hasPostTags = db_table_exists('post_tags');

if ($hasPostTags) {
  $rows = DB::run(
    "SELECT t.*,
            COALESCE(pt.cnt,0) AS total_posts
       FROM tags t
       LEFT JOIN (
         SELECT tag_id, COUNT(*) AS cnt
           FROM post_tags
       GROUP BY tag_id
       ) pt ON pt.tag_id = t.id
   ORDER BY t.name ASC"
  )->fetchAll();
} else {
  // Fallback: table not present → still list tags, counts as 0
  $rows = DB::run("SELECT t.*, 0 AS total_posts FROM tags t ORDER BY t.name ASC")->fetchAll();
}

include __DIR__ . '/partials/header.php';
include __DIR__ . '/partials/sidebar.php';
?>
<div class="admin-container">
  <h2>Tag</h2>
  <?php if (isset($_GET['saved'])): ?>
    <div class="alert success">Saved.</div>
  <?php endif; ?>

  <?php if (!$hasPostTags): ?>
    <div class="alert warn" style="margin-bottom:12px">
      <strong>Note:</strong> The <code>post_tags</code> table was not found. Tag post counts will show as 0.
      When ready, create it like:<br>
      <code>
        CREATE TABLE post_tags (
          post_id INT UNSIGNED NOT NULL,
          tag_id  INT UNSIGNED NOT NULL,
          PRIMARY KEY (post_id, tag_id),
          INDEX idx_post_tags_tag (tag_id),
          CONSTRAINT fk_pt_post FOREIGN KEY (post_id) REFERENCES posts(id) ON DELETE CASCADE,
          CONSTRAINT fk_pt_tag  FOREIGN KEY (tag_id)  REFERENCES tags(id)  ON DELETE CASCADE
        );
      </code>
    </div>
  <?php endif; ?>

  <div class="grid-2">
    <!-- Form -->
    <div class="panel">
      <h3>Tag Form</h3>
      <form method="post" autocomplete="off">
        <label>Name *</label>
        <input name="name" required placeholder="Ex. Breaking, Sports, Politics etc." value="<?= e($tag['name']) ?>">

        <label>Slug (optional)</label>
        <input name="slug" placeholder="auto from name if empty" value="<?= e($tag['slug']) ?>">

        <div style="display:flex; gap:10px; align-items:center; margin-top:8px">
          <button class="btn"><?= $id ? 'Update' : 'Save' ?></button>
          <?php if ($id): ?>
            <a class="btn" href="tag.php">New</a>
          <?php endif; ?>
        </div>
      </form>
    </div>

    <!-- List -->
    <div class="panel">
      <h3>All Tags</h3>
      <table class="table">
        <thead>
          <tr>
            <th>SL</th>
            <th>Name</th>
            <th>Total Posts</th>
            <th>Manage</th>
          </tr>
        </thead>
        <tbody>
          <?php if ($rows): $i=1; foreach ($rows as $r): ?>
            <tr>
              <td><?= $i++ ?></td>
              <td><?= e($r['name']) ?></td>
              <td><?= (int)$r['total_posts'] ?></td>
              <td>
                <a class="btn-sm" href="tag.php?id=<?= (int)$r['id'] ?>">Edit</a>
                <a class="btn-sm" style="background:#f44336"
                   href="tag.php?del=<?= (int)$r['id'] ?>"
                   onclick="return confirm('Delete this tag?')">Del</a>
              </td>
            </tr>
          <?php endforeach; else: ?>
            <tr><td colspan="4">No tags yet.</td></tr>
          <?php endif; ?>
        </tbody>
      </table>
    </div>
  </div>
</div>
<?php include __DIR__ . '/partials/footer.php'; ?>
