<?php
// admin/section_position_list.php
// 1) When NO ?section_id: show the "Homepage List" (sections with preview + actions)
// 2) When ?section_id=ID: show the item manager for that section (move up/down, delete, etc.)

require_once __DIR__ . '/../core/auth.php';
require_login();
require_role(['admin']);

require_once __DIR__ . '/../core/db.php';
require_once __DIR__ . '/../core/helpers.php';

/* ---------------- helpers ---------------- */
function table_exists(string $tbl): bool {
  return (int) DB::run(
    "SELECT COUNT(*) FROM information_schema.TABLES
     WHERE TABLE_SCHEMA = DATABASE() AND TABLE_NAME=?",
    [$tbl]
  )->fetchColumn() > 0;
}
function db_col_exists(string $table, string $col): bool {
  return (int) DB::run(
    "SELECT COUNT(*) FROM information_schema.COLUMNS
     WHERE TABLE_SCHEMA=DATABASE() AND TABLE_NAME=? AND COLUMN_NAME=?",
    [$table,$col]
  )->fetchColumn() > 0;
}

/** Try to locate a preview image by slug. Put your previews in one of:
 *  - admin/assets/section_previews/SLUG.(png|jpg|webp)
 *  - admin/assets/sections/SLUG.(png|jpg|webp)
 * Returns web path or a tiny placeholder SVG.
 */
function section_preview_src(string $slug): string {
  $candsDir = [
    __DIR__ . '/assets/section_previews',
    __DIR__ . '/assets/sections',
  ];
  $candsExt = ['png','jpg','jpeg','webp'];

  foreach ($candsDir as $dir) {
    foreach ($candsExt as $ext) {
      $fs = $dir . '/' . $slug . '.' . $ext;
      if (is_file($fs)) {
        // convert to web path relative to admin/
        $rel = str_replace(__DIR__ . '/', '', $fs);
        return $rel;
      }
    }
  }

  // fallback inline SVG (lightweight placeholder)
  $svg = 'data:image/svg+xml;utf8,' . rawurlencode(
    '<svg xmlns="http://www.w3.org/2000/svg" width="520" height="300">
       <rect width="100%" height="100%" fill="#f4f7fb"/>
       <text x="50%" y="50%" dominant-baseline="middle" text-anchor="middle"
             font-family="Arial" font-size="20" fill="#99a3b2">No preview</text>
     </svg>'
  );
  return $svg;
}

/* ---------------- guards ---------------- */
if (!table_exists('home_sections') || !table_exists('home_section_items')) {
  include __DIR__ . '/partials/header.php';
  include __DIR__ . '/partials/sidebar.php';
  echo '<div class="admin-container"><div class="panel"><h3>Homepage List</h3>
        <div class="alert error">Tables <code>home_sections</code> and/or <code>home_section_items</code> not found.</div>
        <p>Run the SQL shown below in phpMyAdmin, then refresh this page.</p></div></div></main></div>';
  include __DIR__ . '/partials/footer.php';
  exit;
}

/* ---------------- actions (list or items) ---------------- */

// LIST actions (delete a section)
if (!isset($_GET['section_id']) && isset($_GET['del'])) {
  $id = (int)$_GET['del'];
  DB::run("DELETE FROM home_sections WHERE id=?",[$id]);
  DB::run("DELETE FROM home_section_items WHERE section_id=?",[$id]);
  header("Location: section_position_list.php?deleted=1");
  exit;
}

/* ---------------- fetch data depending on mode ---------------- */
$section_id = isset($_GET['section_id']) ? (int)$_GET['section_id'] : 0;

/* ---------------- include header/sidebar ---------------- */
include __DIR__ . '/partials/header.php';
include __DIR__ . '/partials/sidebar.php';

/* =======================================================================
   MODE 1: LIST (no section_id)  -> "Homepage List"
   ======================================================================= */
if (!$section_id):
  $hasRows = db_col_exists('home_sections','rows_count');
  $hasCols = db_col_exists('home_sections','cols_count');

  $sections = DB::run(
    "SELECT s.*,
            (SELECT COUNT(*) FROM home_section_items i WHERE i.section_id=s.id) AS items_count
       FROM home_sections s
   ORDER BY s.position, s.id"
  )->fetchAll();
?>
<style>
  .hp-list .preview {
    border:2px solid #16a34a; /* green */
    border-radius:8px;
    padding:6px;
    display:inline-block;
    background:#fff;
  }
  .hp-list .preview img { display:block; max-width:520px; height:auto; }
  .badge {
    display:inline-block; padding:2px 6px; font-size:12px; line-height:1;
    border-radius:4px; background:#111827; color:#fff; margin-right:6px;
  }
  .btn-ghost{ display:inline-block; padding:6px 10px; border-radius:6px; background:#0f172a; color:#fff; text-decoration:none;}
  .btn-ghost:hover{ opacity:.9}
  .btn-edit{ background:#16a34a;}
  .btn-del{ background:#ef4444;}
</style>

<div class="admin-container hp-list">
  <h2>Homepage List</h2>
  <?php if (isset($_GET['deleted'])): ?><div class="alert success">Section removed.</div><?php endif; ?>

  <table class="table">
    <thead>
      <tr>
        <th style="width:80px">ID</th>
        <th>Category/Name (file_name)</th>
        <th style="width:560px">Image</th>
        <th style="width:220px">Action</th>
      </tr>
    </thead>
    <tbody>
    <?php foreach ($sections as $s):
      $img = section_preview_src($s['slug'] ?? '');
      $rows = $hasRows ? (int)$s['rows_count'] : null;
      $cols = $hasCols ? (int)$s['cols_count'] : null;
    ?>
      <tr>
        <td><?= (int)$s['id'] ?></td>

        <td>
          <div style="font-weight:600; font-size:18px"><?= e($s['name']) ?></div>
          <div style="color:#6b7280; font-size:13px">
            (<?= e($s['slug']) ?>)
            <?php if ($rows !== null): ?> · rows: <?= $rows ?><?php endif; ?>
            <?php if ($cols !== null): ?> · cols: <?= $cols ?><?php endif; ?>
          </div>
        </td>

        <td>
          <span class="preview">
            <img src="<?= e($img) ?>" alt="<?= e($s['slug'] ?? 'preview') ?>">
          </span>
        </td>

        <td>
          <span class="badge" title="Items in section"><?= (int)$s['items_count'] ?></span>
          <span class="badge" title="Active (1/0)"><?= (int)$s['is_active'] ?></span>
          <div style="margin-top:8px; display:flex; gap:8px; flex-wrap:wrap">
            <a class="btn-ghost" href="section_position_list.php?section_id=<?= (int)$s['id'] ?>">Manage</a>
            <a class="btn-ghost btn-edit" href="section_index.php?id=<?= (int)$s['id'] ?>">Edit</a>
            <a class="btn-ghost btn-del"
               href="section_position_list.php?del=<?= (int)$s['id'] ?>"
               onclick="return confirm('Delete this section and its items?')">Delete</a>
          </div>
        </td>
      </tr>
    <?php endforeach; if (!$sections): ?>
      <tr><td colspan="4">No sections yet. Create one from <a href="section_index.php">Section index</a>.</td></tr>
    <?php endif; ?>
    </tbody>
  </table>
</div>

<?php
/* =======================================================================
   MODE 2: ITEMS (with ?section_id) -> reuse the manager UI
   ======================================================================= */
else:
  // --- Item manager (same behavior you asked earlier) ---

  // section info
  $sec = DB::run("SELECT * FROM home_sections WHERE id=?",[$section_id])->fetch();
  if (!$sec) {
    echo '<div class="admin-container"><div class="panel"><h3>Section not found.</h3></div></div></main></div>';
    include __DIR__ . '/partials/footer.php'; exit;
  }

  // add
  if ($_SERVER['REQUEST_METHOD']==='POST' && isset($_POST['__add'])) {
    $post_id = (int)($_POST['post_id'] ?? 0);
    $position = trim($_POST['position'] ?? '');
    if ($post_id) {
      if ($position==='') {
        $position = (int)DB::run("SELECT COALESCE(MAX(position),0)+1 FROM home_section_items WHERE section_id=?",[$section_id])->fetchColumn();
      } else $position=(int)$position;
      DB::run("INSERT INTO home_section_items(section_id,post_id,position) VALUES (?,?,?)",[$section_id,$post_id,$position]);
      header("Location: section_position_list.php?section_id=".$section_id."&saved=1"); exit;
    }
  }

  // delete
  if (isset($_GET['del'])) {
    DB::run("DELETE FROM home_section_items WHERE id=?",[(int)$_GET['del']]);
    header("Location: section_position_list.php?section_id=".$section_id); exit;
  }

  // move
  if (isset($_GET['move'], $_GET['id'])) {
    $id  = (int)$_GET['id'];
    $dir = $_GET['move']==='up' ? 'up' : 'down';
    $cur = DB::run("SELECT id, position FROM home_section_items WHERE id=? AND section_id=?",[$id,$section_id])->fetch();
    if ($cur) {
      if ($dir==='up') {
        $nbr = DB::run("SELECT id, position FROM home_section_items
                        WHERE section_id=? AND position < ?
                        ORDER BY position DESC LIMIT 1",[$section_id,$cur['position']])->fetch();
      } else {
        $nbr = DB::run("SELECT id, position FROM home_section_items
                        WHERE section_id=? AND position > ?
                        ORDER BY position ASC LIMIT 1",[$section_id,$cur['position']])->fetch();
      }
      if ($nbr) {
        DB::run("UPDATE home_section_items SET position=? WHERE id=?",[$nbr['position'],$cur['id']]);
        DB::run("UPDATE home_section_items SET position=? WHERE id=?",[$cur['position'],$nbr['id']]);
      }
    }
    header("Location: section_position_list.php?section_id=".$section_id); exit;
  }

  // normalize
  if (isset($_GET['fix'])) {
    $rows = DB::run("SELECT id FROM home_section_items WHERE section_id=? ORDER BY position, id",[$section_id])->fetchAll();
    $n=1; foreach($rows as $r){ DB::run("UPDATE home_section_items SET position=? WHERE id=?",[$n++,(int)$r['id']]); }
    header("Location: section_position_list.php?section_id=".$section_id."&fixed=1"); exit;
  }

  $titleCol = db_col_exists('posts','title_bn') ? 'p.title_bn'
            : (db_col_exists('posts','title') ? 'p.title' : "NULL");

  $items = DB::run(
    "SELECT i.id,i.post_id,i.position, $titleCol AS post_title
       FROM home_section_items i
  LEFT JOIN posts p ON p.id=i.post_id
      WHERE i.section_id=?
   ORDER BY i.position, i.id",
    [$section_id]
  )->fetchAll();
?>
<div class="admin-container">
  <h2>Manage Items — <?= e($sec['name']) ?> <small style="color:#6b7280"> (<?= e($sec['slug']) ?>)</small></h2>
  <?php if (isset($_GET['saved'])): ?><div class="alert success">Item added.</div><?php endif; ?>
  <?php if (isset($_GET['fixed'])): ?><div class="alert success">Positions normalized.</div><?php endif; ?>

  <div class="grid-2">
    <div class="panel">
      <h3>Add Item</h3>
      <form method="post">
        <input type="hidden" name="__add" value="1">
        <label>Post ID</label>
        <input type="number" name="post_id" min="1" required placeholder="e.g. 123">

        <label>Position <small>(blank = append)</small></label>
        <input type="number" name="position" min="1" placeholder="auto">

        <button class="btn">Add</button>
        <a class="btn" href="section_position_list.php">← Back to list</a>
        <a class="btn" href="section_position_list.php?section_id=<?= (int)$section_id ?>&fix=1">Normalize</a>
      </form>
    </div>

    <div class="panel">
      <h3>Items</h3>
      <table class="table">
        <thead><tr>
          <th style="width:70px">#</th><th style="width:120px">Post ID</th><th>Title</th>
          <th style="width:120px">Position</th><th style="width:190px">Action</th>
        </tr></thead>
        <tbody>
        <?php if ($items): $i=1; foreach($items as $it): ?>
          <tr>
            <td><?= $i++ ?></td>
            <td><?= (int)$it['post_id'] ?></td>
            <td><?= e($it['post_title'] ?? '') ?></td>
            <td><?= (int)$it['position'] ?></td>
            <td>
              <a class="btn-sm" href="section_position_list.php?section_id=<?= (int)$section_id ?>&id=<?= (int)$it['id'] ?>&move=up">⬆️</a>
              <a class="btn-sm" href="section_position_list.php?section_id=<?= (int)$section_id ?>&id=<?= (int)$it['id'] ?>&move=down">⬇️</a>
              <a class="btn-sm" style="background:#f44336"
                 href="section_position_list.php?section_id=<?= (int)$section_id ?>&del=<?= (int)$it['id'] ?>"
                 onclick="return confirm('Delete this item?')">Delete</a>
            </td>
          </tr>
        <?php endforeach; else: ?>
          <tr><td colspan="5">No items. Add a post from the left form.</td></tr>
        <?php endif; ?>
        </tbody>
      </table>
    </div>
  </div>
</div>
<?php endif; // end modes ?>

</main></div>
<?php include __DIR__ . '/partials/footer.php'; ?>
