<?php
// admin/photo.php — Photo Gallery list (backward-compatible with older schemas)

// Auth
require_once __DIR__ . '/../core/auth.php';
require_login();
require_role(['admin']);

// Core
require_once __DIR__ . '/../core/db.php';
require_once __DIR__ . '/../core/helpers.php';

$SELF = basename(__FILE__);

/* ---------- helpers ---------- */
function table_exists($t) {
  return (int) DB::run(
    "SELECT COUNT(*) FROM information_schema.TABLES
     WHERE TABLE_SCHEMA = DATABASE() AND TABLE_NAME = ?", [$t]
  )->fetchColumn() > 0;
}
function column_exists($t, $c) {
  return (int) DB::run(
    "SELECT COUNT(*) FROM information_schema.COLUMNS
     WHERE TABLE_SCHEMA = DATABASE() AND TABLE_NAME = ? AND COLUMN_NAME = ?",
    [$t, $c]
  )->fetchColumn() > 0;
}

/* ---------- ensure table + migrate columns (idempotent) ---------- */
if (!table_exists('photos')) {
  DB::run("
    CREATE TABLE photos (
      id INT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
      title VARCHAR(255) NOT NULL,
      image_path VARCHAR(255) NOT NULL,
      caption TEXT NULL,
      is_active TINYINT(1) NOT NULL DEFAULT 1,
      created_at TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP,
      updated_at TIMESTAMP NULL DEFAULT NULL ON UPDATE CURRENT_TIMESTAMP,
      KEY idx_active (is_active),
      KEY idx_created (created_at)
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4
  ");
} else {
  $hasImagePath = column_exists('photos', 'image_path');
  $hasImage     = column_exists('photos', 'image');

  // If old schema used `image`, rename to `image_path`
  if (!$hasImagePath && $hasImage) {
    DB::run("ALTER TABLE photos CHANGE image image_path VARCHAR(255) NOT NULL");
    $hasImagePath = true;
    $hasImage     = false; // it was renamed
  }
  // If neither exists (very old/empty table), add canonical column
  if (!$hasImagePath && !$hasImage) {
    DB::run("ALTER TABLE photos ADD COLUMN image_path VARCHAR(255) NOT NULL AFTER title");
    $hasImagePath = true;
  }
  if (!column_exists('photos', 'caption')) {
    DB::run("ALTER TABLE photos ADD COLUMN caption TEXT NULL AFTER image_path");
  }
  if (!column_exists('photos', 'is_active')) {
    DB::run("ALTER TABLE photos ADD COLUMN is_active TINYINT(1) NOT NULL DEFAULT 1");
  }
  if (!column_exists('photos', 'created_at')) {
    DB::run("ALTER TABLE photos ADD COLUMN created_at TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP");
  }
  if (!column_exists('photos', 'updated_at')) {
    DB::run("ALTER TABLE photos ADD COLUMN updated_at TIMESTAMP NULL DEFAULT NULL ON UPDATE CURRENT_TIMESTAMP");
  }
}

/* ---------- know which column to read/write for images ---------- */
$imgColPath = column_exists('photos', 'image_path') ? 'image_path' :
              (column_exists('photos', 'image') ? 'image' : null);

/* ---------- actions ---------- */
if (isset($_GET['toggle'])) {
  $id = (int) $_GET['toggle'];
  DB::run("UPDATE photos SET is_active = 1 - is_active WHERE id = ?", [$id]);
  header("Location: {$SELF}");
  exit;
}

if (isset($_GET['del'])) {
  $id = (int) $_GET['del'];

  // Try to remove the physical file too (paths in DB are relative to /admin)
  if ($imgColPath) {
    $img = DB::run("SELECT {$imgColPath} FROM photos WHERE id = ?", [$id])->fetchColumn();
    if ($img) {
      $disk = __DIR__ . '/' . ltrim($img, '/');
      if (is_file($disk)) { @unlink($disk); }
    }
  }

  DB::run("DELETE FROM photos WHERE id = ?", [$id]);
  header("Location: {$SELF}");
  exit;
}

/* ---------- data (build query without referencing missing columns) ---------- */
$imgExpr = $imgColPath ? $imgColPath : "''";
$sql = "
  SELECT
    id,
    title,
    caption,
    is_active,
    created_at,
    {$imgExpr} AS img
  FROM photos
  ORDER BY id DESC
";
$rows = DB::run($sql)->fetchAll(PDO::FETCH_ASSOC);

/* ---------- UI ---------- */
include __DIR__ . '/partials/header.php';
include __DIR__ . '/partials/sidebar.php';
?>
<style>
.badge{display:inline-block;padding:4px 10px;border-radius:999px;font-size:12px}
.badge.on{background:#10b981;color:#fff}
.badge.off{background:#ef4444;color:#fff}
.btn{display:inline-block;background:#0f172a;color:#fff;border:none;border-radius:10px;padding:10px 14px;text-decoration:none}
.btn-chip{display:inline-block;padding:6px 10px;border-radius:10px;background:#0f172a;color:#fff;text-decoration:none}
.btn-chip.secondary{background:#111827}
.panel{background:#fff;border:1px solid #e5e7eb;border-radius:12px;padding:18px}
.table .thumb{width:70px;height:50px;object-fit:cover;border-radius:6px;border:1px solid #e5e7eb}
.actions a{display:inline-block;margin-right:8px}
</style>

<div class="admin-container">
  <div style="display:flex;align-items:center;justify-content:space-between;gap:12px;margin-bottom:10px">
    <h2>All Photo</h2>
    <a class="btn" href="create.php">+ New Photo</a>
  </div>

  <div class="panel">
    <table class="table">
      <thead>
        <tr>
          <th>SL</th>
          <th>Picture</th>
          <th>Title</th>
          <th>Status</th>
          <th>Uploaded at</th>
          <th>Manage</th>
        </tr>
      </thead>
      <tbody>
      <?php if ($rows): $i = 1; foreach ($rows as $r): ?>
        <tr>
          <td><?= $i++ ?></td>
          <td>
            <?php if (!empty($r['img'])): ?>
              <img class="thumb" src="<?= e($r['img']) ?>" alt="">
            <?php endif; ?>
          </td>
          <td>
            <div style="font-weight:600"><?= e($r['title']) ?></div>
            <?php if (!empty($r['caption'])): ?>
              <div style="color:#6b7280;font-size:13px;max-width:520px"><?= e($r['caption']) ?></div>
            <?php endif; ?>
          </td>
          <td>
            <span class="badge <?= $r['is_active'] ? 'on' : 'off' ?>">
              <?= $r['is_active'] ? 'On' : 'Off' ?>
            </span>
          </td>
          <td><?= e(date('d M Y', strtotime($r['created_at']))) ?></td>
          <td class="actions">
            <a class="btn-chip" href="<?= e($SELF) ?>?toggle=<?= (int)$r['id'] ?>">
              Turn <?= $r['is_active'] ? 'Off' : 'On' ?>
            </a>
            <a class="btn-chip secondary" href="javascript:void(0)"
               onclick="if(confirm('Delete this photo?')) location.href='<?= e($SELF) ?>?del=<?= (int)$r['id'] ?>'">
              Delete
            </a>
          </td>
        </tr>
      <?php endforeach; else: ?>
        <tr><td colspan="6">No photos yet.</td></tr>
      <?php endif; ?>
      </tbody>
    </table>
  </div>
</div>

<?php include __DIR__ . '/partials/footer.php';
