<?php
// admin/media.php — Unified Media Library (images, video, audio, docs)
// - Shows files from multiple upload roots
// - Multi upload (drag & drop), copy URL, open, rename, delete
// - Reuses existing `photos` table (file/url + alt) when present

// ------------ Auth ------------
require_once __DIR__ . '/../core/auth.php';
require_login();
require_role(['admin','editor']);

// ------------ Helpers / DB ------------
require_once __DIR__ . '/../core/db.php';
require_once __DIR__ . '/../core/helpers.php';

// ------------ Upload roots ------------
// You can add/remove roots to match your project.
$UPLOAD_ROOTS = [];

// admin/uploads (always)
$adminUploads = __DIR__ . '/uploads';
$UPLOAD_ROOTS[] = ['dir' => $adminUploads, 'web' => 'uploads'];

// public/uploads (optional)
$publicUploads = realpath(__DIR__ . '/../public/uploads');
if ($publicUploads && is_dir($publicUploads)) {
  // from /admin/, path to public/uploads is ../public/uploads
  $UPLOAD_ROOTS[] = ['dir' => $publicUploads, 'web' => '../public/uploads'];
}

// ../uploads (optional at project root)
$rootUploads = realpath(__DIR__ . '/../uploads');
if ($rootUploads && is_dir($rootUploads)) {
  $UPLOAD_ROOTS[] = ['dir' => $rootUploads, 'web' => '../uploads'];
}

/* Default upload target where new files go (you can change this) */
$DEFAULT_TARGET_DIR  = $adminUploads . '/media';
$DEFAULT_TARGET_WEB  = 'uploads/media';

// Ensure default target exists
if (!is_dir($DEFAULT_TARGET_DIR)) {
  @mkdir($DEFAULT_TARGET_DIR, 0775, true);
}

// ---------------- photos table helpers ----------------
function table_has(string $table): bool {
  return (int)DB::run(
    "SELECT COUNT(*) FROM information_schema.tables
     WHERE table_schema=DATABASE() AND table_name=?",
    [$table]
  )->fetchColumn() > 0;
}
function col_has(string $table, string $col): bool {
  return (int)DB::run(
    "SELECT COUNT(*) FROM information_schema.columns
     WHERE table_schema=DATABASE() AND table_name=? AND column_name=?",
    [$table, $col]
  )->fetchColumn() > 0;
}
$PHOTOS_EXISTS = table_has('photos');

// Decide URL column in `photos`
function photos_url_col(): ?string {
  if (!table_has('photos')) return null;
  if (col_has('photos','file')) return 'file';
  if (col_has('photos','url'))  return 'url';
  return null;
}
function photos_ensure_cols() {
  // Ensure at least one URL column exists (prefer `file`), and add `alt` if missing when needed
  if (!table_has('photos')) return;
  if (!col_has('photos','file') && !col_has('photos','url')) {
    DB::run("ALTER TABLE photos ADD COLUMN file VARCHAR(255) NULL");
  }
  // do not force add alt until we need it on insert (we'll handle conditionally)
}
function photos_insert(string $url, ?string $alt=null) {
  if (!table_has('photos')) return;
  photos_ensure_cols();
  $col = photos_url_col() ?? 'file';
  $hasAlt = col_has('photos','alt');
  if ($alt !== null && !$hasAlt) {
    DB::run("ALTER TABLE photos ADD COLUMN alt VARCHAR(255) NULL");
    $hasAlt = true;
  }
  if ($hasAlt) {
    DB::run("INSERT INTO photos($col, alt) VALUES(?,?)", [$url, $alt]);
  } else {
    DB::run("INSERT INTO photos($col) VALUES(?)", [$url]);
  }
}
function photos_update_url(string $oldUrl, string $newUrl) {
  if (!table_has('photos')) return;
  $col = photos_url_col(); if (!$col) return;
  DB::run("UPDATE photos SET $col=? WHERE $col=?", [$newUrl, $oldUrl]);
}
function photos_delete_by_url(string $url) {
  if (!table_has('photos')) return;
  $col = photos_url_col(); if (!$col) return;
  DB::run("DELETE FROM photos WHERE $col=?", [$url]);
}

// --------------- misc utilities ---------------
if (!function_exists('str_starts_with')) {
  function str_starts_with($haystack, $needle) {
    return (string)$needle === '' || strncmp($haystack, $needle, strlen($needle)) === 0;
  }
}
function _normalize_path($p){ return str_replace('\\','/', $p); }

function _allowed_path(string $abs, array $roots): bool {
  $absN = realpath($abs) ?: $abs;
  $absN = _normalize_path($absN);
  foreach ($roots as $r) {
    $root = realpath($r['dir']);
    if (!$root) continue;
    $root = _normalize_path($root);
    if (str_starts_with($absN, $root)) return true;
  }
  return false;
}
function _human_bytes($b){
  $u=['B','KB','MB','GB','TB']; $i=0;
  while($b>=1024 && $i<count($u)-1){ $b/=1024; $i++; }
  return sprintf('%.1f %s',$b,$u[$i]);
}
function _is_img($ext){ return in_array($ext, ['jpg','jpeg','png','webp','gif','svg'], true); }
function _is_vid($ext){ return in_array($ext, ['mp4','webm','ogv','mov'], true); }
function _is_aud($ext){ return in_array($ext, ['mp3','wav','ogg'], true); }

// Convert absolute path to a web URL using $UPLOAD_ROOTS mapping
function abs_to_url(string $abs, array $roots): ?string {
  $absN = _normalize_path(realpath($abs) ?: $abs);
  foreach ($roots as $r) {
    $base = _normalize_path(realpath($r['dir']));
    if (!$base) continue;
    if (str_starts_with($absN, $base)) {
      $rel = ltrim(substr($absN, strlen($base)), '/');
      return rtrim($r['web'],'/') . '/' . $rel;
    }
  }
  return null;
}

$msg = '';

// --------------- Delete (GET) ---------------
if (isset($_GET['del'])) {
  $abs = base64_decode((string)$_GET['del'], true);
  if ($abs && _allowed_path($abs, $UPLOAD_ROOTS) && is_file($abs)) {
    // capture old URL to clean photos row too
    $oldUrl = abs_to_url($abs, $UPLOAD_ROOTS);
    if (@unlink($abs)) {
      if ($oldUrl) photos_delete_by_url($oldUrl);
      $msg = 'Deleted: ' . e(basename($abs));
    } else {
      $msg = 'Delete failed.';
    }
  } else {
    $msg = 'Invalid file.';
  }
}

// --------------- Rename (POST) ---------------
if ($_SERVER['REQUEST_METHOD'] === 'POST' && ($_POST['act'] ?? '') === 'rename') {
  if (!csrf_check($_POST['csrf'] ?? '')) die('Bad CSRF token');

  $abs = base64_decode((string)($_POST['p'] ?? ''), true);
  $new = trim($_POST['newname'] ?? '');
  if (!$abs || !$new) {
    $msg = 'Missing data for rename.';
  } elseif (!_allowed_path($abs, $UPLOAD_ROOTS) || !is_file($abs)) {
    $msg = 'Invalid file.';
  } else {
    if (preg_match('~[\\\\/]+~', $new)) {
      $msg = 'New name cannot contain slashes.';
    } else {
      $dir  = dirname($abs);
      $safe = preg_replace('~[^a-zA-Z0-9._-]+~', '_', $new);
      if ($safe === '') $safe = 'file';
      $target = $dir . DIRECTORY_SEPARATOR . $safe;

      $oldUrl = abs_to_url($abs, $UPLOAD_ROOTS);
      if (@rename($abs, $target)) {
        $newUrl = abs_to_url($target, $UPLOAD_ROOTS);
        if ($oldUrl && $newUrl) photos_update_url($oldUrl, $newUrl);
        $msg = 'Renamed to: ' . e($safe);
      } else {
        $msg = 'Rename failed.';
      }
    }
  }
}

// --------------- Upload (POST, multiple) ---------------
if ($_SERVER['REQUEST_METHOD'] === 'POST' && ($_POST['act'] ?? '') === 'upload') {
  if (!csrf_check($_POST['csrf'] ?? '')) die('Bad CSRF token');

  $destDir = $DEFAULT_TARGET_DIR;
  $destWeb = $DEFAULT_TARGET_WEB;

  $allowed = [
    'jpg','jpeg','png','webp','gif','svg',
    'mp4','webm','ogv','mov',
    'mp3','wav','ogg',
    'pdf','txt','csv','zip','rar',
    'doc','docx','xls','xlsx','ppt','pptx'
  ];
  $done = [];
  $altAll = trim($_POST['alt'] ?? '') ?: null;

  if (!empty($_FILES['files']['name']) && is_array($_FILES['files']['name'])) {
    foreach ($_FILES['files']['name'] as $i => $orig) {
      if ($_FILES['files']['error'][$i] !== UPLOAD_ERR_OK) continue;
      $ext = strtolower(pathinfo($orig, PATHINFO_EXTENSION));
      if ($ext && !in_array($ext, $allowed, true)) continue;

      $safe = preg_replace('~[^a-zA-Z0-9._-]+~','_', $orig);
      $base = pathinfo($safe, PATHINFO_FILENAME);
      $ext2 = strtolower(pathinfo($safe, PATHINFO_EXTENSION));
      try { $rand = bin2hex(random_bytes(2)); } catch(\Throwable $e) { $rand = dechex(mt_rand(0,65535)); }
      $uniq = $base . '-' . date('Ymd-His') . '-' . $rand;
      $final = $uniq . ($ext2 ? '.' . $ext2 : '');
      $abs   = $destDir . DIRECTORY_SEPARATOR . $final;

      if (move_uploaded_file($_FILES['files']['tmp_name'][$i], $abs)) {
        $done[] = $final;

        // Save into `photos` if present
        $url = rtrim($destWeb,'/') . '/' . $final;
        photos_insert($url, $altAll);
      }
    }
  }
  $msg = $done ? ('Uploaded: ' . e(implode(', ', $done))) : 'No files were uploaded.';
}

// --------------- Collect files from all roots ---------------
$files = [];
$pushFile = function(string $rootDir, string $webBase, string $absPath) use (&$files){
  if (!is_file($absPath)) return;
  $rel = ltrim(str_replace([$rootDir . DIRECTORY_SEPARATOR, $rootDir . '/'], '', $absPath), '\\/');
  $url = rtrim($webBase, '/').'/'.str_replace(DIRECTORY_SEPARATOR, '/', $rel);
  $files[] = [
    'abs'   => $absPath,
    'url'   => $url,
    'name'  => basename($absPath),
    'size'  => filesize($absPath),
    'mtime' => filemtime($absPath),
    'ext'   => strtolower(pathinfo($absPath, PATHINFO_EXTENSION)),
  ];
};

$scan = function(string $dir, callable $cb) use (&$scan){
  if (!is_dir($dir)) return;
  $it = @scandir($dir);
  if (!$it) return;
  foreach ($it as $f) {
    if ($f === '.' || $f === '..') continue;
    $abs = $dir . DIRECTORY_SEPARATOR . $f;
    if (is_dir($abs)) {
      $scan($abs, $cb);
    } else {
      $cb($abs);
    }
  }
};

foreach ($UPLOAD_ROOTS as $root) {
  $base = $root['dir'];
  $web  = $root['web'];
  if (!is_dir($base)) continue;
  $scan($base, fn($abs) => $pushFile($base, $web, $abs));
}
usort($files, fn($a,$b) => $b['mtime'] <=> $a['mtime']);

// --------------- UI chrome ---------------
include __DIR__ . '/partials/header.php';
include __DIR__ . '/partials/sidebar.php';
?>
<style>
.toolbar{display:flex;align-items:center;justify-content:space-between;gap:12px;margin-bottom:16px}
.btn{display:inline-block;background:#0f172a;color:#fff;border:none;border-radius:10px;padding:10px 14px;text-decoration:none;cursor:pointer}
.btn.secondary{background:#1f2937}
.notice{background:#ecfdf5;color:#064e3b;border:1px solid #a7f3d0;padding:10px 12px;border-radius:8px;margin-bottom:14px}
.panel{background:#fff;border:1px solid #e5e7eb;border-radius:12px;padding:18px}
.grid{display:grid;grid-template-columns:repeat(auto-fill,minmax(200px,1fr));gap:14px}
.card{border:1px solid #e5e7eb;border-radius:12px;overflow:hidden;background:#fff;display:flex;flex-direction:column}
.thumb{height:140px;background:#f8fafc;display:flex;align-items:center;justify-content:center}
.thumb img,.thumb video,.thumb audio{max-height:140px;max-width:100%}
.card .meta{padding:10px 12px;display:flex;flex-direction:column;gap:6px}
.card .name{font-weight:600;white-space:nowrap;overflow:hidden;text-overflow:ellipsis}
.card .sub{color:#6b7280;font-size:12px}
.card .actions{display:flex;gap:8px;margin-top:6px;flex-wrap:wrap}
.card .chip{font-size:12px;padding:6px 8px;border-radius:8px;border:1px solid #e5e7eb;background:#fff;cursor:pointer;text-decoration:none;color:#111827}
.card .chip.danger{background:#fee2e2;border-color:#fecaca;color:#7f1d1d}
.drag{border:2px dashed #cbd5e1;border-radius:12px;padding:14px;text-align:center;background:#f8fafc}
.drag.dragover{background:#eef2ff;border-color:#93c5fd}
.search{display:flex;gap:8px}
.search input{border:1px solid #d1d5db;border-radius:10px;padding:8px 10px;width:280px}
.badge{display:inline-block;padding:2px 8px;border-radius:999px;background:#eef2ff;color:#1d4ed8;font-size:11px}
.small{font-size:12px;color:#6b7280}
</style>

<div class="admin-container">

  <div class="toolbar">
    <h2>Media Library</h2>
    <div class="search">
      <input id="mediaSearch" type="search" placeholder="Search file name…">
      <a class="btn" href="#" onclick="document.getElementById('fileInput').click();return false;">+ Add New</a>
    </div>
  </div>

  <?php if (!empty($msg)): ?>
    <div class="notice"><?= e($msg) ?></div>
  <?php endif; ?>

  <div class="panel" style="margin-bottom:16px">
    <form id="uploadForm" method="post" enctype="multipart/form-data">
      <input type="hidden" name="csrf" value="<?= e(csrf_token()) ?>">
      <input type="hidden" name="act" value="upload">
      <input id="fileInput" type="file" name="files[]" multiple hidden>
      <div id="dropZone" class="drag">
        <strong>Drag & drop</strong> files here or click <u>Add New</u> above.
        <div class="small" style="margin-top:6px">
          Shows files from <code>admin/uploads/**</code>, <code>public/uploads/**</code>, and <code>../uploads/**</code>.
          New uploads go to <code><?= e($DEFAULT_TARGET_WEB) ?></code>.
        </div>
      </div>
      <div style="margin-top:8px;display:flex;gap:8px;align-items:center">
        <label class="small">ALT (optional)</label>
        <input type="text" name="alt" placeholder="ALT text for all selected files"
               style="border:1px solid #d1d5db;border-radius:8px;padding:6px 10px;flex:1">
      </div>
      <div id="pendingNames" style="margin-top:10px;color:#374151;display:none"></div>
      <div style="margin-top:10px;display:flex;gap:8px">
        <button class="btn" type="submit">Upload</button>
        <button class="btn secondary" type="button"
          onclick="document.getElementById('fileInput').value='';document.getElementById('pendingNames').style.display='none';document.getElementById('pendingNames').innerHTML=''">Clear</button>
      </div>
    </form>
  </div>

  <div class="panel">
    <?php if (!$files): ?>
      <p style="margin:0">No media yet. Use <strong>Add New</strong> to upload.</p>
    <?php else: ?>
      <div id="mediaGrid" class="grid">
        <?php foreach ($files as $f): ?>
          <div class="card" data-name="<?= e(strtolower($f['name'])) ?>">
            <div class="thumb">
              <?php if (_is_img($f['ext'])): ?>
                <img src="<?= e($f['url']) ?>" alt="">
              <?php elseif (_is_vid($f['ext'])): ?>
                <video src="<?= e($f['url']) ?>" preload="metadata" muted></video>
              <?php elseif (_is_aud($f['ext'])): ?>
                <audio src="<?= e($f['url']) ?>" preload="metadata" controls></audio>
              <?php else: ?>
                <div>
                  <div style="font-size:40px;line-height:1.1">📄</div>
                  <div class="badge"><?= strtoupper(e($f['ext'] ?: 'FILE')) ?></div>
                </div>
              <?php endif; ?>
            </div>
            <div class="meta">
              <div class="name" title="<?= e($f['name']) ?>"><?= e($f['name']) ?></div>
              <div class="sub"><?= e(_human_bytes($f['size'])) ?> • <?= e(date('d M Y, H:i', $f['mtime'])) ?></div>
              <div class="actions">
                <button class="chip" data-url="<?= e($f['url']) ?>" onclick="copyUrl(this)" type="button">Copy URL</button>
                <a class="chip" href="<?= e($f['url']) ?>" target="_blank">Open</a>
                <button class="chip" type="button"
                        data-p="<?= e(base64_encode($f['abs'])) ?>"
                        data-name="<?= e($f['name']) ?>"
                        onclick="renameFile(this)">Rename</button>
                <a class="chip danger"
                   href="media.php?del=<?= urlencode(base64_encode($f['abs'])) ?>"
                   onclick="return confirm('Delete this file?')">Delete</a>
              </div>
            </div>
          </div>
        <?php endforeach; ?>
      </div>
    <?php endif; ?>
  </div>
</div>

<form id="actionForm" method="post" style="display:none">
  <input type="hidden" name="csrf" value="<?= e(csrf_token()) ?>">
  <input type="hidden" name="act"  value="">
  <input type="hidden" name="p"    value="">
  <input type="hidden" name="newname" value="">
</form>

<script>
// Copy resolved absolute URL to clipboard
function copyUrl(btn){
  const url = btn.getAttribute('data-url');
  const a = document.createElement('a');
  a.href = url; // browser resolves to absolute
  const abs = a.href;
  navigator.clipboard.writeText(abs)
    .then(()=>{ btn.textContent='Copied!'; setTimeout(()=>btn.textContent='Copy URL',1200); })
    .catch(()=>{ btn.textContent='Failed';  setTimeout(()=>btn.textContent='Copy URL',1200); });
}

// Rename (prompt -> post)
function renameFile(btn){
  const p = btn.getAttribute('data-p');
  const current = btn.getAttribute('data-name') || '';
  const next = prompt('New file name (no slashes):', current);
  if (!next || next === current) return;
  const f = document.getElementById('actionForm');
  f.act.value = 'rename';
  f.p.value = p;
  f.newname.value = next.trim();
  f.submit();
}

// Drag & drop upload UX
const drop = document.getElementById('dropZone');
const input = document.getElementById('fileInput');
const pending = document.getElementById('pendingNames');

['dragenter','dragover'].forEach(evt=>{
  drop.addEventListener(evt,(e)=>{ e.preventDefault(); e.stopPropagation(); drop.classList.add('dragover'); });
});
['dragleave','drop'].forEach(evt=>{
  drop.addEventListener(evt,(e)=>{ e.preventDefault(); e.stopPropagation(); drop.classList.remove('dragover'); });
});
drop.addEventListener('drop',(e)=>{
  input.files = e.dataTransfer.files;
  showPending();
});
input.addEventListener('change', showPending);

function showPending(){
  if (!input.files || !input.files.length){ pending.style.display='none'; pending.innerHTML=''; return; }
  let names = Array.from(input.files).map(f=>f.name).join(', ');
  pending.textContent = 'Selected: ' + names;
  pending.style.display='block';
}

// Client-side search
const search = document.getElementById('mediaSearch');
const grid   = document.getElementById('mediaGrid');
if (search && grid){
  search.addEventListener('input', ()=>{
    const q = search.value.trim().toLowerCase();
    grid.querySelectorAll('.card').forEach(card=>{
      const hit = card.getAttribute('data-name').includes(q);
      card.style.display = hit ? '' : 'none';
    });
  });
}
</script>

<?php include __DIR__ . '/partials/footer.php';
