<?php
// admin/dashboard.php — schema-safe, collapsible left menu, all existing features intact

require_once __DIR__ . '/../core/auth.php';
require_login();
require_role(['admin']); // এডিটর চাইলে: ['admin','editor']

require_once __DIR__ . '/../core/db.php';
require_once __DIR__ . '/../core/helpers.php';

/* ========== helpers ========== */
function db_col_exists(string $table, string $col): bool {
  $row = DB::run(
    "SELECT COUNT(*) FROM information_schema.COLUMNS
     WHERE TABLE_SCHEMA = DATABASE() AND TABLE_NAME=? AND COLUMN_NAME=?",
    [$table,$col]
  )->fetchColumn();
  return (int)$row > 0;
}
$cur = basename($_SERVER['SCRIPT_NAME']);
$in = fn(array $files) => in_array($cur, $files, true);

/* ========== Date helpers ========== */
$today0 = date('Y-m-d 00:00:00');
$today1 = date('Y-m-d 23:59:59');
$y0     = date('Y-m-d 00:00:00', strtotime('-1 day'));
$y1     = date('Y-m-d 23:59:59', strtotime('-1 day'));
$m0     = date('Y-m-01 00:00:00');
$m1     = date('Y-m-t 23:59:59');

/* ========== Top counters ========== */
$catHasIsActive = db_col_exists('categories','is_active');
$counts = [
  'posts_total'     => (int) DB::run("SELECT COUNT(*) FROM posts")->fetchColumn(),
  'posts_published' => (int) DB::run("SELECT COUNT(*) FROM posts WHERE status='published'")->fetchColumn(),
  'posts_scheduled' => (int) DB::run("SELECT COUNT(*) FROM posts WHERE status='scheduled'")->fetchColumn(),
  'users'           => (int) DB::run("SELECT COUNT(*) FROM users")->fetchColumn(),
  'categories'      => (int) (
    $catHasIsActive
      ? DB::run("SELECT COUNT(*) FROM categories WHERE is_active=1")->fetchColumn()
      : DB::run("SELECT COUNT(*) FROM categories")->fetchColumn()
  ),
  'pv_today'        => (int) DB::run("SELECT COUNT(*) FROM pageviews WHERE created_at BETWEEN ? AND ?", [$today0,$today1])->fetchColumn(),
  'pv_yesterday'    => (int) DB::run("SELECT COUNT(*) FROM pageviews WHERE created_at BETWEEN ? AND ?", [$y0,$y1])->fetchColumn(),
  'pv_month'        => (int) DB::run("SELECT COUNT(*) FROM pageviews WHERE created_at BETWEEN ? AND ?", [$m0,$m1])->fetchColumn(),
  'posts_today'     => (int) DB::run(
    "SELECT COUNT(*) FROM posts
      WHERE status='published'
        AND (publish_at BETWEEN ? AND ?
             OR (publish_at IS NULL AND created_at BETWEEN ? AND ?))",
    [$today0,$today1,$today0,$today1]
  )->fetchColumn(),
];

/* ========== SEO Progress (schema-safe) ========== */
$hasSeoT = db_col_exists('posts','seo_title');
$hasSeoD = db_col_exists('posts','seo_desc');
$hasSeoK = db_col_exists('posts','seo_keywords');

$seo = ['total'=>$counts['posts_total'],'ok'=>0,'pct'=>null,'note'=>''];
if ($seo['total']>0 && ($hasSeoT||$hasSeoD||$hasSeoK)) {
  $conds = [];
  if ($hasSeoT) $conds[] = "NULLIF(TRIM(seo_title),'') IS NOT NULL";
  if ($hasSeoD) $conds[] = "NULLIF(TRIM(seo_desc),'') IS NOT NULL";
  if ($hasSeoK) $conds[] = "NULLIF(TRIM(seo_keywords),'') IS NOT NULL";
  $where = implode(' OR ', $conds);
  $seo['ok']  = (int) DB::run("SELECT COUNT(*) FROM posts WHERE ($where)")->fetchColumn();
  $seo['pct'] = (int) round(($seo['ok'] / max(1,$seo['total']))*100);
} else {
  $seo['note'] = 'No SEO fields found';
}

/* ========== Recent posts & leaderboard (schema-safe) ========== */
$hasAuthorId = db_col_exists('posts','author_id');

$recent = ($hasAuthorId
  ? DB::run(
      "SELECT p.id,p.title_bn,p.status,p.publish_at,c.name_bn AS cat,u.name AS author
         FROM posts p
    LEFT JOIN categories c ON c.id=p.category_id
    LEFT JOIN users u       ON u.id=p.author_id
     ORDER BY COALESCE(p.publish_at,p.created_at) DESC
        LIMIT 10"
    )->fetchAll()
  : DB::run(
      "SELECT p.id,p.title_bn,p.status,p.publish_at,c.name_bn AS cat,NULL AS author
         FROM posts p
    LEFT JOIN categories c ON c.id=p.category_id
     ORDER BY COALESCE(p.publish_at,p.created_at) DESC
        LIMIT 10"
    )->fetchAll()
);

$leaderboard = ($hasAuthorId
  ? DB::run(
      "SELECT COALESCE(u.name,'(unknown)') AS name, COUNT(*) AS total
         FROM posts p
    LEFT JOIN users u ON u.id=p.author_id
        WHERE p.status='published' AND p.created_at BETWEEN ? AND ?
     GROUP BY u.id
     ORDER BY total DESC
        LIMIT 12",
      [date('Y-m-d 00:00:00', strtotime('-7 days')), $today1]
    )->fetchAll()
  : DB::run(
      "SELECT '(All Editors)' AS name, COUNT(*) AS total
         FROM posts
        WHERE status='published' AND created_at BETWEEN ? AND ?
        LIMIT 1",
      [date('Y-m-d 00:00:00', strtotime('-7 days')), $today1]
    )->fetchAll()
);

/* ========== Last week visitors & scheduled queue ========== */
$week = DB::run(
  "SELECT DATE(created_at) AS d, COUNT(*) AS v
     FROM pageviews
    WHERE created_at >= DATE_SUB(CURDATE(), INTERVAL 6 DAY)
 GROUP BY DATE(created_at)
 ORDER BY d ASC"
)->fetchAll();
$weekLabels = array_map(fn($r)=>$r['d'],$week);
$weekData   = array_map(fn($r)=>(int)$r['v'],$week);

$queue = DB::run(
  "SELECT id,title_bn,publish_at
     FROM posts
    WHERE status='scheduled'
 ORDER BY publish_at ASC
    LIMIT 10"
)->fetchAll();

include __DIR__ . '/partials/header.php';
?>
<style>
/* ======= Left sidebar (collapsible) ======= */
.admin-shell{display:flex;gap:18px}
.sidenav{
  width:240px;background:#1f2630;color:#cfd3da;border-radius:10px;margin-top:6px;
  position:sticky;top:12px;height:fit-content;box-shadow:0 2px 8px rgba(0,0,0,.05)
}
.sidenav .brand{font-weight:700;color:#fff;padding:12px 16px;border-bottom:1px solid rgba(255,255,255,.07)}
.sidenav .grp{border-bottom:1px solid rgba(255,255,255,.05)}
.sidenav .head{
  display:flex;align-items:center;gap:10px;width:100%;padding:12px 16px;background:none;border:0;
  color:#cfd3da;cursor:pointer;text-align:left;font-weight:600
}
.sidenav .caret{margin-left:auto;transition:transform .18s ease}
.sidenav .open .caret{transform:rotate(90deg)}
.sidenav a.item{
  display:flex;align-items:center;gap:10px;padding:10px 36px;color:#cfd3da;text-decoration:none
}
.sidenav a.item:hover{background:#2b3440;color:#fff}
.sidenav a.active{background:#2b3440;color:#fff}
.sidenav .items{display:none;padding-bottom:8px}
.sidenav .open .items{display:block}
.admin-main{flex:1}

/* small bits */
.seo-bar{height:10px;background:#e9eef4;border-radius:8px;overflow:hidden}
.seo-bar > span{display:block;height:100%;background:#22c55e;width:0}
.seo-note{font-size:12px;color:#777;margin-top:6px}
</style>

<div class="admin-shell">
  <!-- LEFT SIDEBAR -->
  <?php
    $openDash  = $in(['dashboard.php']);
    $openPost  = $in(['posts.php','post_edit.php']);
    $openMedia = $in(['media.php']);
    $openUsers = $in(['users.php']);

    // OPEN state for the big Configuration bucket (includes *all* config pages)
    $openConf  = $in([
      'category.php','categories.php', // handle either filename
      'city.php','country.php','designation.php','division.php',
      'tag.php','upazila.php',
      'menu_setup.php','section_index.php','section_position_list.php','trending_tag.php'
    ]);

    // Ad Clients
    $openClients = $in(['ad_clients.php','ad_client_edit.php']);
  ?>
  <aside class="sidenav" aria-label="Sidebar navigation">
    <div class="brand">ভোরের আকাশ</div>

    <!-- Dashboard -->
    <div class="grp <?= $openDash?'open':'' ?>">
      <button class="head" type="button">
        <span>📊 Dashboard</span><span class="caret">▸</span>
      </button>
      <div class="items">
        <a class="item <?= $cur==='dashboard.php'?'active':'' ?>" href="dashboard.php">Dashboard home</a>
      </div>
    </div>

    <!-- User Role -->
    <div class="grp <?= $openUsers?'open':'' ?>">
      <button class="head" type="button">
        <span>👥 User Role Management</span><span class="caret">▸</span>
      </button>
      <div class="items">
        <a class="item <?= $cur==='users.php'?'active':'' ?>" href="users.php">Users</a>
      </div>
    </div>

    <!-- Configuration (ALL items) -->
    <div class="grp <?= $openConf?'open':'' ?>">
      <button class="head" type="button">
        <span>⚙️ Configuration</span><span class="caret">▸</span>
      </button>
      <div class="items">
        <?php
          // Smart link: use categories.php if it exists, otherwise category.php
          $catPage = file_exists(__DIR__.'/categories.php') ? 'categories.php' : (file_exists(__DIR__.'/category.php') ? 'category.php' : 'categories.php');
        ?>
        <a class="item <?= in_array($cur,['categories.php','category.php'],true)?'active':'' ?>" href="<?= $catPage ?>">Category</a>

        <a class="item <?= $cur==='city.php'?'active':'' ?>" href="city.php">City</a>
        <a class="item <?= $cur==='country.php'?'active':'' ?>" href="country.php">Country</a>
        <a class="item <?= $cur==='designation.php'?'active':'' ?>" href="designation.php">Designation</a>
        <a class="item <?= $cur==='division.php'?'active':'' ?>" href="division.php">Division</a>
        <a class="item <?= $cur==='tag.php'?'active':'' ?>" href="tag.php">Tag</a>
        <a class="item <?= $cur==='upazila.php'?'active':'' ?>" href="upazila.php">Upazila</a>
        <a class="item <?= $cur==='menu_setup.php'?'active':'' ?>" href="menu_setup.php">Menu Setup</a>
        <a class="item <?= $cur==='section_index.php'?'active':'' ?>" href="section_index.php">Section index</a>
        <a class="item <?= $cur==='section_position_list.php'?'active':'' ?>" href="section_position_list.php">Section Position List</a>
        <a class="item <?= $cur==='trending_tag.php'?'active':'' ?>" href="trending_tag.php">Trending Tag</a>
      </div>
    </div>

    <!-- Post -->
    <div class="grp <?= $openPost?'open':'' ?>">
      <button class="head" type="button">
        <span>📝 Post</span><span class="caret">▸</span>
      </button>
      <div class="items">
        <a class="item <?= $cur==='post_edit.php'?'active':'' ?>" href="post_edit.php">New Post</a>
        <a class="item <?= $cur==='posts.php'?'active':'' ?>" href="posts.php">Post List</a>
      </div>
    </div>

    <!-- Ad Client -->
    <div class="grp <?= $openClients?'open':'' ?>">
      <button class="head" type="button"><span>🏷️ Ad. Client</span><span class="caret">▸</span></button>
      <div class="items">
        <a class="item <?= $cur==='ad_client_edit.php'?'active':'' ?>" href="ad_client_edit.php">New Client</a>
        <a class="item <?= $cur==='ad_clients.php'?'active':'' ?>" href="ad_clients.php">Client List</a>
      </div>
    </div>

    <!-- Advertisement (placeholder) -->
    <div class="grp">
      <button class="head" type="button"><span>📢 Advertisement</span><span class="caret">▸</span></button>
      <div class="items">
	   <a class="item <?= $cur==='adsenses.php'?'active':'' ?>" href="adsenses.php">
        Adsense List</a>
      </div>
    </div>

    <!-- Photo Gallery (placeholder) -->
    <div class="grp">
      <button class="head" type="button"><span>🖼️ Photo Gallery</span><span class="caret">▸</span></button>
      <div class="items">
        <a class="item <?= $cur==='photo.php'?'active':'' ?>" href="photo.php">photo gallery</a>
      </div>
    </div>

    <!-- Video Gallery (placeholder) -->
    <div class="grp">
      <button class="head" type="button"><span>🎬 Video Gallery</span><span class="caret">▸</span></button>
      <div class="items">
         <a class="item <?= $cur==='photo.php'?'active':'' ?>" href="video-category.php">Video Gallery</a>
         <a class="item <?= $cur==='photo.php'?'active':'' ?>" href="video.php">Video List</a>
      </div>
    </div>

    <!-- Website -->
    <div class="grp <?= $openMedia?'open':'' ?>">
      <button class="head" type="button"><span>🌐 Website</span><span class="caret">▸</span></button>
      <div class="items">
        <a class="item <?= $cur==='media.php'?'active':'' ?>" href="media.php">Media</a>
        <a class="item" href="../" target="_blank">Visit site</a>
      </div>
    </div>

    <!-- Employee (placeholder) -->
    <div class="grp">
      <button class="head" type="button"><span>🧑‍💼 Employee</span><span class="caret">▸</span></button>
      <div class="items">
        <a class="item" href="#">Department List</a>
      </div>
    </div>
   <div class="mini"><strong>Pages</strong><br>
        <a class="btn-sm" href="page-edit.php">Create</a>
        <a class="btn-sm" href="pages.php">Manage</a>
   </div>
   <div class="mini"><strong>Appearance</strong><br>
      <a class="btn-sm" href="appearance-header.php">Header</a>
      <a class="btn-sm" href="appearance-footer.php">Footer</a>
    </div>
  <div class="mini"><strong>Media</strong><br>
    <a class="btn-sm" href="media.php">Library</a>
  </div>

    <div class="grp">
      <a class="item" href="logout.php">🚪 Logout</a>
    </div>
  </aside>

  <!-- MAIN -->
  <main class="admin-main">
    <div class="admin-container">
      <h2>Vurrer Kagoj</h2>

      <div class="stats-cards">
        <div class="card"><div>Posts</div><strong><?= $counts['posts_total'] ?></strong></div>
        <div class="card"><div>Published</div><strong><?= $counts['posts_published'] ?></strong></div>
        <div class="card"><div>Scheduled</div><strong><?= $counts['posts_scheduled'] ?></strong></div>
        <div class="card"><div>Users</div><strong><?= $counts['users'] ?></strong></div>
        <div class="card"><div>Categories</div><strong><?= $counts['categories'] ?></strong></div>
      </div>

      <!-- SEO -->
      <div class="panel" style="margin:16px 0">
        <h3>SEO Progress</h3>
        <?php if ($seo['pct'] !== null): ?>
          <div class="mini-cards" style="margin-bottom:8px">
            <div class="mini"><div>Total Posts</div><strong><?= (int)$seo['total'] ?></strong></div>
            <div class="mini"><div>SEO-ready Posts</div><strong><?= (int)$seo['ok'] ?></strong></div>
            <div class="mini"><div>Completion</div><strong><?= (int)$seo['pct'] ?>%</strong></div>
          </div>
          <div class="seo-bar"><span style="width:<?= max(0,min(100,(int)$seo['pct'])) ?>%"></span></div>
          <div class="seo-note">A post counted as SEO-ready if it has any of: <?= $hasSeoT?'seo_title':'' ?><?= ($hasSeoT && ($hasSeoD||$hasSeoK))?', ':'' ?><?= $hasSeoD?'seo_desc':'' ?><?= ($hasSeoD && $hasSeoK)?', ':'' ?><?= $hasSeoK?'seo_keywords':'' ?>.</div>
        <?php else: ?>
          <div class="seo-note"><em><?= e($seo['note']) ?></em></div>
        <?php endif; ?>
      </div>

      <div class="grid-2">
        <div class="panel">
          <h3>Today / Yesterday / This Month</h3>
          <div class="mini-cards">
            <div class="mini"><div>Today’s Posts</div><strong><?= $counts['posts_today'] ?></strong></div>
            <div class="mini"><div>Today’s Visitors</div><strong><?= $counts['pv_today'] ?></strong></div>
            <div class="mini"><div>Yesterday Visitors</div><strong><?= $counts['pv_yesterday'] ?></strong></div>
            <div class="mini"><div>This Month Visitors</div><strong><?= $counts['pv_month'] ?></strong></div>
          </div>
          <canvas id="chartWeek" height="120"></canvas>
        </div>

        <div class="panel">
          <h3>Quick Actions</h3>
          <a class="btn" href="post_edit.php">+ New Post</a>
          <a class="btn" href="<?= $catPage ?>">+ New Category</a>
          <a class="btn" href="../core/scheduler.php" target="_blank" rel="noopener">Run Scheduler</a>

          <h3 style="margin-top:16px">Scheduled Queue</h3>
          <table class="table">
            <thead><tr><th>Time</th><th>Title</th></tr></thead>
            <tbody>
              <?php foreach ($queue as $q): ?>
                <tr>
                  <td><?= e($q['publish_at']) ?></td>
                  <td><a href="post_edit.php?id=<?= (int)$q['id'] ?>"><?= e($q['title_bn']) ?></a></td>
                </tr>
              <?php endforeach; if (!$queue): ?>
                <tr><td colspan="2">No items.</td></tr>
              <?php endif; ?>
            </tbody>
          </table>
        </div>
      </div>

      <div class="grid-2">
        <div class="panel">
          <h3>Today’s News Editor Post (last 10)</h3>
          <table class="table">
            <thead><tr><th>#</th><th>Title</th><th>Cat</th><th>Status</th><th>Time</th><th>Action</th></tr></thead>
            <tbody>
              <?php $i=1; foreach ($recent as $r): ?>
                <tr>
                  <td><?= $i++ ?></td>
                  <td><?= e(mb_strimwidth($r['title_bn'] ?? '', 0, 60, '…', 'UTF-8')) ?></td>
                  <td><?= e($r['cat'] ?? '') ?></td>
                  <td><?= e($r['status'] ?? '') ?></td>
                  <td><?= e($r['publish_at'] ?? '') ?></td>
                  <td><a class="btn-sm" href="post_edit.php?id=<?= (int)$r['id'] ?>">View</a></td>
                </tr>
              <?php endforeach; if (!$recent): ?>
                <tr><td colspan="6">No data.</td></tr>
              <?php endif; ?>
            </tbody>
          </table>
        </div>

        <div class="panel">
          <h3>Last Week Visitor</h3>
          <table class="table">
            <thead><tr><th>Day</th><th>Visits</th></tr></thead>
            <tbody>
              <?php foreach ($week as $w): ?>
                <tr><td><?= e($w['d']) ?></td><td><?= (int)$w['v'] ?></td></tr>
              <?php endforeach; if (!$week): ?>
                <tr><td colspan="2">No data.</td></tr>
              <?php endif; ?>
            </tbody>
          </table>

          <h3 style="margin-top:16px">Editor Leaderboard (7d)</h3>
          <table class="table">
            <thead><tr><th>SL</th><th>Name</th><th>Total News</th></tr></thead>
            <tbody>
              <?php $i=1; foreach ($leaderboard as $row): ?>
                <tr><td><?= $i++ ?></td><td><?= e($row['name']) ?></td><td><?= (int)$row['total'] ?></td></tr>
              <?php endforeach; if (!$leaderboard): ?>
                <tr><td colspan="3">No data.</td></tr>
              <?php endif; ?>
            </tbody>
          </table>
        </div>
      </div>
    </div>
  </main>
</div>

<link rel="stylesheet" href="assets/css/admin.css">
<script src="https://cdn.jsdelivr.net/npm/chart.js"></script>
<script>
// collapsible groups
document.querySelectorAll('.sidenav .head').forEach(btn=>{
  btn.addEventListener('click',()=>btn.parentElement.classList.toggle('open'));
});
const ctx = document.getElementById('chartWeek');
if (ctx) {
  new Chart(ctx, {
    type:'line',
    data:{labels:<?= json_encode($weekLabels, JSON_UNESCAPED_UNICODE) ?>,datasets:[{label:'Visitors',data:<?= json_encode($weekData) ?>,tension:.3}]},
    options:{plugins:{legend:{display:false}},scales:{y:{beginAtZero:true}}}
  });
}
</script>
<?php include __DIR__ . '/partials/footer.php'; ?>
