<?php
// admin/create.php — Photo Gallery: create a new photo

require_once __DIR__ . '/../core/auth.php';
require_login();
require_role(['admin']);

require_once __DIR__ . '/../core/db.php';
require_once __DIR__ . '/../core/helpers.php';

/* ---------- list page (auto-detect) ---------- */
if (file_exists(__DIR__ . '/photo.php')) {
  $LIST_PAGE = 'photo.php';
} elseif (file_exists(__DIR__ . '/photos.php')) {
  $LIST_PAGE = 'photos.php';
} elseif (file_exists(__DIR__ . '/phot.php')) {
  $LIST_PAGE = 'phot.php';
} else {
  // last resort, keep phot.php to avoid a hard 404
  $LIST_PAGE = 'phot.php';
}

/* ---------- small helpers ---------- */
function table_exists($t) {
  return (int) DB::run(
    "SELECT COUNT(*) FROM information_schema.TABLES
     WHERE TABLE_SCHEMA = DATABASE() AND TABLE_NAME = ?",
    [$t]
  )->fetchColumn() > 0;
}
function column_exists($t, $c) {
  return (int) DB::run(
    "SELECT COUNT(*) FROM information_schema.COLUMNS
     WHERE TABLE_SCHEMA = DATABASE() AND TABLE_NAME = ? AND COLUMN_NAME = ?",
    [$t, $c]
  )->fetchColumn() > 0;
}

/* ---------- ensure table + columns ---------- */
if (!table_exists('photos')) {
  DB::run("
    CREATE TABLE photos (
      id INT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
      title VARCHAR(255) NOT NULL,
      image_path VARCHAR(255) NOT NULL,
      is_active TINYINT(1) NOT NULL DEFAULT 1,
      created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4
  ");
} else {
  // migrate old schemas safely
  $hasImagePath = column_exists('photos', 'image_path');
  $hasImage     = column_exists('photos', 'image');

  if (!$hasImagePath && $hasImage) {
    DB::run("ALTER TABLE photos CHANGE image image_path VARCHAR(255) NOT NULL");
    $hasImagePath = true;
  }
  if (!$hasImagePath) {
    DB::run("ALTER TABLE photos ADD COLUMN image_path VARCHAR(255) NOT NULL AFTER title");
  }
  if (!column_exists('photos', 'is_active')) {
    DB::run("ALTER TABLE photos ADD COLUMN is_active TINYINT(1) NOT NULL DEFAULT 1");
  }
  if (!column_exists('photos', 'created_at')) {
    DB::run("ALTER TABLE photos ADD COLUMN created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP");
  }
}

/* ---------- handle submit ---------- */
$errors = [];
$title  = trim($_POST['title'] ?? '');

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
  if ($title === '') {
    $errors[] = 'Title is required.';
  }

  if (!isset($_FILES['photo']) || $_FILES['photo']['error'] === UPLOAD_ERR_NO_FILE) {
    $errors[] = 'Please choose an image.';
  } else {
    $allowed = ['jpg','jpeg','png','webp'];
    $ext = strtolower(pathinfo($_FILES['photo']['name'], PATHINFO_EXTENSION));
    if (!in_array($ext, $allowed, true)) {
      $errors[] = 'Only JPG, PNG or WEBP images are allowed.';
    }
    if ($_FILES['photo']['error'] !== UPLOAD_ERR_OK) {
      $errors[] = 'Upload failed. Please try again.';
    }
  }

  if (!$errors) {
    // /admin/uploads/photos
    $uploadDir = __DIR__ . '/uploads/photos';
    if (!is_dir($uploadDir)) {
      @mkdir($uploadDir, 0775, true);
    }

    $slug = preg_replace('~[^a-z0-9]+~i', '-', $title);
    $slug = trim($slug, '-');
    $ext  = strtolower(pathinfo($_FILES['photo']['name'], PATHINFO_EXTENSION));
    $name = $slug . '-' . date('Ymd-His') . '-' . bin2hex(random_bytes(2)) . '.' . $ext;

    $absPath = $uploadDir . '/' . $name;
    $relPath = 'uploads/photos/' . $name; // stored in DB (relative from admin/)

    if (!move_uploaded_file($_FILES['photo']['tmp_name'], $absPath)) {
      $errors[] = 'Could not move uploaded file.';
    } else {
      DB::run(
        "INSERT INTO photos (title, image_path, is_active) VALUES (?,?,1)",
        [$title, $relPath]
      );
      header("Location: {$LIST_PAGE}?saved=1");
      exit;
    }
  }
}

/* ---------- view ---------- */
include __DIR__ . '/partials/header.php';
include __DIR__ . '/partials/sidebar.php';
?>
<style>
  .panel{background:#fff;border:1px solid #e5e7eb;border-radius:12px;padding:18px}
  .form-row{margin-bottom:14px}
  .form-row label{display:block;font-weight:600;margin-bottom:6px}
  .form-row input[type=text], .form-row input[type=file]{width:100%;padding:10px;border:1px solid #d1d5db;border-radius:8px;background:#fff}
  .btn{display:inline-block;background:#16a34a;color:#fff;border:none;border-radius:10px;padding:10px 18px;cursor:pointer}
  .btn-secondary{background:#0f172a;color:#fff}
  .alert{padding:10px 12px;border-radius:8px;margin-bottom:12px}
  .alert.error{background:#fee2e2;color:#7f1d1d;border:1px solid #fecaca}
</style>

<div class="admin-container">
  <div style="display:flex;justify-content:space-between;align-items:center;margin-bottom:16px">
    <h2>Photo Category Form</h2>
    <a class="btn btn-secondary" href="<?= e($LIST_PAGE) ?>">All Photo's</a>
  </div>

  <?php if ($errors): ?>
    <div class="alert error"><?= e(implode(' ', $errors)) ?></div>
  <?php endif; ?>

  <div class="panel">
    <form method="post" enctype="multipart/form-data" novalidate>
      <div class="form-row">
        <label>Title</label>
        <input type="text" name="title" placeholder="Ex. Banglar Green." value="<?= e($title) ?>" required>
      </div>

      <div class="form-row">
        <label>Display Picture</label>
        <input type="file" name="photo" accept=".jpg,.jpeg,.png,.webp" required>
      </div>

      <button class="btn">Save</button>
      <a class="btn btn-secondary" href="<?= e($LIST_PAGE) ?>" style="margin-left:8px">Back</a>
    </form>
  </div>
</div>

<?php include __DIR__ . '/partials/footer.php';
