<?php
// admin/appearance-header.php  OR  admin/appearance-footer.php
// Unified builder (rows → columns → widgets) for both Header and Footer.

require_once __DIR__.'/../core/auth.php';  require_login();  require_role(['admin']);
require_once __DIR__.'/../core/helpers.php';
require_once __DIR__.'/../core/options.php';

if (!function_exists('e')) { function e($s){ return htmlspecialchars((string)$s, ENT_QUOTES, 'UTF-8'); } }

/* Detect context from our filename */
$WHAT   = (strpos(basename(__FILE__), 'footer') !== false) ? 'footer' : 'header';
$OPTKEY = $WHAT . '_layout';

/* Save */
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
  $json = $_POST['layout_json'] ?? '[]';
  $data = json_decode($json, true);
  if (!is_array($data)) $data = [];
  option_set($OPTKEY, $data);
  header('Location: '.basename(__FILE__).'?saved=1');
  exit;
}

/* Load */
$layout = option_get($OPTKEY, []);
?>
<!doctype html>
<meta charset="utf-8">
<title><?= e(ucfirst($WHAT)) ?> Builder</title>
<link rel="stylesheet" href="assets/css/admin.css">
<link rel="stylesheet" href="assets/css/admin-wide.css">

<div class="admin-container admin-shell">
  <?php $top = __DIR__.'/partials/topbar.php'; if (is_file($top)) include $top; ?>

  <div class="panel">
    <div style="display:flex;justify-content:space-between;align-items:center">
      <h2 style="margin:0"><?= e(ucfirst($WHAT)) ?> Builder</h2>
      <div>
        <a class="btn" href="pages.php">Pages</a>
        <a class="btn" href="appearance-header.php">Header</a>
        <a class="btn" href="appearance-footer.php">Footer</a>
      </div>
    </div>

    <?php if (!empty($_GET['saved'])): ?>
      <div class="notice"><?= e(ucfirst($WHAT)) ?> saved.</div>
    <?php endif; ?>

    <form method="post" id="builderForm">
      <input type="hidden" name="layout_json" id="layout_json">

      <div class="toolbar" style="display:flex;gap:8px;margin:8px 0 12px">
        <button type="button" class="btn" id="addRow">+ Row</button>
        <button type="button" class="btn secondary" id="preview">Preview layout JSON</button>
        <button class="btn" style="margin-left:auto">Save</button>
      </div>

      <div id="canvas" class="canvas"></div>

      <div class="toolbar" style="display:flex;gap:8px;margin-top:14px">
        <button class="btn">Save</button>
        <a class="btn" href="<?= e(basename(__FILE__)) ?>">Reset</a>
      </div>
    </form>
  </div>
</div>

<style>
  .canvas{ display:flex; flex-direction:column; gap:14px }
  .row{ border:1px dashed #cbd5e1; border-radius:12px; padding:10px; background:#fafafa }
  .row-head{ display:flex; align-items:center; justify-content:space-between; margin-bottom:8px }
  .row-cols{ display:flex; gap:10px; flex-wrap:wrap }
  .col{ background:#fff; border:1px solid #e5e7eb; border-radius:10px; padding:10px; flex:1 1 48% }
  .col-head{ display:flex; gap:8px; align-items:center; margin-bottom:8px; flex-wrap:wrap }
  .col .widgets{ display:flex; gap:8px; flex-wrap:wrap; min-height:42px }
  .w{ border:1px solid #d1d5db; background:#fff; padding:6px 8px; border-radius:8px; display:flex; align-items:center; gap:8px }
  .w .grab{ cursor:grab }
  .w .cfg{ width:100% }
  .chip{ font-size:12px; padding:4px 8px; border:1px solid #e5e7eb; border-radius:999px; background:#fff; cursor:pointer }
  .btn-danger{ background:#b91c1c; color:#fff }
  .muted{ color:#666 }
  .small{ font-size:12px }
  .sep{ width:6px; display:inline-block }
  .inline{ display:inline-flex; align-items:center; gap:6px }
</style>

<script>
/* ====== Initial data (safe JSON from PHP arrays) ====== */
const initial = <?php
  if (!empty($layout) && is_array($layout)) {
    echo json_encode($layout, JSON_UNESCAPED_UNICODE);
  } else {
    if ($WHAT === 'header') {
      $default = [[
        'columns' => [
          ['width'=>35,'spread'=>false,'widgets'=>[
            ['type'=>'logo','cfg'=>['image'=>'/uploads/logo.png','link'=>'/']]
          ]],
          ['width'=>65,'spread'=>true,'widgets'=>[
            ['type'=>'menu','cfg'=>[]],
            ['type'=>'search','cfg'=>['placeholder'=>'Search…']],
            ['type'=>'lang','cfg'=>['items'=>[['label'=>'EN','href'=>'/en']]]]
          ]]
        ]
      ]];
    } else {
      $default = [[
        'columns' => [
          ['width'=>33,'spread'=>false,'widgets'=>[
            ['type'=>'html','cfg'=>['html'=>'<p>&copy; '.date('Y').' Vorer Akash. All rights reserved.</p>']]
          ]],
          ['width'=>34,'spread'=>false,'widgets'=>[
            ['type'=>'menu','cfg'=>[]]
          ]],
          ['width'=>33,'spread'=>false,'widgets'=>[
            ['type'=>'social','cfg'=>['links'=>[['label'=>'Facebook','url'=>'#'],['label'=>'YouTube','url'=>'#']]]]
          ]]
        ]
      ]];
    }
    echo json_encode($default, JSON_UNESCAPED_UNICODE);
  }
?>;

/* Available widgets */
const WIDGETS = [
  {type:'logo',    label:'Logo'},
  {type:'menu',    label:'Menu'},
  {type:'search',  label:'Search'},
  {type:'lang',    label:'Language'},
  {type:'social',  label:'Social'},
  {type:'html',    label:'HTML'},
  {type:'spacer',  label:'Spacer'},
  {type:'latest',  label:'Latest (post/category)'} // optional
];

/* Helpers */
const el  = (html)=>{const t=document.createElement('template'); t.innerHTML=html.trim(); return t.content.firstChild};
const $id = (id)=>document.getElementById(id);
const canvas = $id('canvas');

/* Toggle helper for the “latest” widget (no closing-tag strings anywhere) */
function setupLatestWidget(host){
  const sel = host.querySelector('.i-source');
  const cat = host.querySelector('.cat-only');
  const pst = host.querySelector('.post-only');
  if (!sel) return;
  const toggle = ()=>{ if (sel.value==='post'){ cat.style.display='none'; pst.style.display='inline-flex'; }
                      else { cat.style.display='inline-flex'; pst.style.display='none'; } };
  toggle();
  sel.addEventListener('change', toggle);
}

/* ====== Render ====== */
function render(){
  canvas.innerHTML = '';
  initial.forEach((row, ri)=>{
    const R = el(`<div class="row" data-ri="${ri}">
      <div class="row-head">
        <strong>Row #${ri+1}</strong>
        <div>
          <button type="button" class="chip addCol">+ Column</button>
          <span class="sep"></span>
          <button type="button" class="chip dupRow">⧉</button>
          <button type="button" class="chip moveUp">↑</button>
          <button type="button" class="chip moveDown">↓</button>
          <button type="button" class="chip btn-danger delRow">Delete</button>
        </div>
      </div>
      <div class="row-cols"></div>
    </div>`);
    const colsWrap = R.querySelector('.row-cols');
    (row.columns||[]).forEach((col, ci)=> colsWrap.appendChild(renderCol(ri, ci, col)));
    canvas.appendChild(R);
  });
  serialize();
}

function renderCol(ri, ci, col){
  const C = el(`<div class="col" data-ri="${ri}" data-ci="${ci}" style="flex-basis:${col.width||100}%">
    <div class="col-head">
      <span class="muted small">Width</span>
      <input type="range" min="10" max="100" value="${col.width||100}" class="wRange">
      <label class="small"><input type="checkbox" class="spread" ${col.spread?'checked':''}> spread</label>
      <button type="button" class="chip addW">+ Widget</button>
      <span class="sep"></span>
      <button type="button" class="chip moveLeft">←</button>
      <button type="button" class="chip moveRight">→</button>
      <button type="button" class="chip dupCol">⧉</button>
      <button type="button" class="chip btn-danger delCol">Delete</button>
    </div>
    <div class="widgets"></div>
  </div>`);
  const wrap = C.querySelector('.widgets');
  (col.widgets||[]).forEach((w, wi)=> wrap.appendChild(renderWidget(ri,ci,wi,w)));
  return C;
}

function renderWidget(ri,ci,wi,w){
  const W = el(`<div class="w" draggable="true" data-ri="${ri}" data-ci="${ci}" data-wi="${wi}">
    <span class="grab">☰</span>
    <select class="type">
      ${WIDGETS.map(x=>`<option value="${x.type}" ${x.type===w.type?'selected':''}>${x.label}</option>`).join('')}
    </select>
    <div class="cfg"></div>
    <button type="button" class="chip btn-danger delW">x</button>
  </div>`);
  injectCfgForm(W.querySelector('.cfg'), w.type, w.cfg||{});
  return W;
}

/* ====== Per-widget forms ====== */
function injectCfgForm(host, type, cfg){
  if(type==='logo'){
    host.innerHTML = `
      <input placeholder="Image URL" class="i-image" value="${cfg.image||''}" style="min-width:220px">
      <input placeholder="Link" class="i-link" value="${cfg.link||'/'}">`;
  } else if(type==='search'){
    host.innerHTML = `<input placeholder="Placeholder" class="i-ph" value="${cfg.placeholder||'Search'}">`;
  } else if(type==='lang'){
    host.innerHTML = `<input placeholder='Items JSON e.g. [{"label":"EN","href":"/en"}]' class="i-json" value="${(JSON.stringify(cfg.items||[{label:'EN',href:'/en'}])).replace(/"/g,'&quot;')}" style="min-width:320px">`;
  } else if(type==='social'){
    host.innerHTML = `<input placeholder='Links JSON e.g. [{"label":"FB","url":"#"}]' class="i-json" value="${(JSON.stringify(cfg.links||[])).replace(/"/g,'&quot;')}" style="min-width:320px">`;
  } else if(type==='html'){
    host.innerHTML = `<input placeholder="Inline HTML" class="i-html" value="${(cfg.html||'').replace(/"/g,'&quot;')}" style="min-width:340px">`;
  } else if(type==='spacer'){
    host.innerHTML = `<span class="muted small">Flexible empty space</span>`;
  } else if(type==='menu'){
    host.innerHTML = `<span class="muted small">Uses <code>site_menus</code> or falls back to categories automatically</span>`;
  } else if(type==='latest'){
    const src   = cfg.source || 'category';
    const id    = cfg.post_id || '';
    const slug  = cfg.cat_slug || '';
    const limit = parseInt(cfg.limit ?? 4, 10);
    const thumbs= !!cfg.thumbs;
    host.innerHTML = `
      <span class="small muted">Source</span>
      <select class="i-source">
        <option value="category" ${src==='category'?'selected':''}>Category</option>
        <option value="post" ${src==='post'?'selected':''}>Post</option>
      </select>
      <span class="sep"></span>
      <span class="inline cat-only">
        <input placeholder="Category slug (e.g. video)" class="i-cat" value="${slug||''}" style="min-width:160px">
        <input type="number" min="1" max="20" class="i-limit" value="${Number.isFinite(limit)?limit:4}" style="width:80px" title="Items">
      </span>
      <span class="inline post-only">
        <input placeholder="Post ID" class="i-postid" value="${id||''}" style="width:120px">
      </span>
      <label class="small inline"><input type="checkbox" class="i-thumbs" ${thumbs?'checked':''}> thumbnails</label>`;
    setupLatestWidget(host);
  }
}

/* ====== Top controls ====== */
$id('addRow').onclick = ()=>{ initial.push({columns:[{width:100,spread:false,widgets:[]} ]}); render(); };

/* ====== Click handlers ====== */
canvas.addEventListener('click', (e)=>{
  const row    = e.target.closest('.row');
  const col    = e.target.closest('.col');
  const widget = e.target.closest('.w');

  // Row
  if (e.target.classList.contains('addCol')) {
    const i=parseInt(row.dataset.ri);
    initial[i].columns = initial[i].columns||[];
    initial[i].columns.push({width:50,spread:false,widgets:[]});
    render(); return;
  }
  if (e.target.classList.contains('dupRow')) {
    const i=parseInt(row.dataset.ri);
    initial.splice(i+1,0,JSON.parse(JSON.stringify(initial[i])));
    render(); return;
  }
  if (e.target.classList.contains('delRow')) {
    initial.splice(parseInt(row.dataset.ri),1);
    if (!initial.length) initial.push({columns:[{width:100,widgets:[]} ]});
    render(); return;
  }
  if (e.target.classList.contains('moveUp')) {
    const i=parseInt(row.dataset.ri); if(i>0){ const t=initial[i]; initial[i]=initial[i-1]; initial[i-1]=t; }
    render(); return;
  }
  if (e.target.classList.contains('moveDown')) {
    const i=parseInt(row.dataset.ri); if(i<initial.length-1){ const t=initial[i]; initial[i]=initial[i+1]; initial[i+1]=t; }
    render(); return;
  }

  // Column
  if (e.target.classList.contains('delCol')) {
    const ri=parseInt(col.dataset.ri), ci=parseInt(col.dataset.ci);
    initial[ri].columns.splice(ci,1);
    if(!initial[ri].columns.length) initial[ri].columns=[{width:100,widgets:[]}];
    render(); return;
  }
  if (e.target.classList.contains('dupCol')) {
    const ri=parseInt(col.dataset.ri), ci=parseInt(col.dataset.ci);
    const clone = JSON.parse(JSON.stringify(initial[ri].columns[ci]));
    initial[ri].columns.splice(ci+1,0,clone);
    render(); return;
  }
  if (e.target.classList.contains('moveLeft')) {
    const ri=parseInt(col.dataset.ri), ci=parseInt(col.dataset.ci);
    if (ci>0){ const arr=initial[ri].columns; const t=arr[ci]; arr[ci]=arr[ci-1]; arr[ci-1]=t; }
    render(); return;
  }
  if (e.target.classList.contains('moveRight')) {
    const ri=parseInt(col.dataset.ri), ci=parseInt(col.dataset.ci);
    const arr=initial[ri].columns;
    if (ci<arr.length-1){ const t=arr[ci]; arr[ci]=arr[ci+1]; arr[ci+1]=t; }
    render(); return;
  }
  if (e.target.classList.contains('addW')) {
    const ri=parseInt(col.dataset.ri), ci=parseInt(col.dataset.ci);
    initial[ri].columns[ci].widgets = initial[ri].columns[ci].widgets||[];
    initial[ri].columns[ci].widgets.push({type:'menu',cfg:{}});
    render(); return;
  }

  // Widget
  if (e.target.classList.contains('delW')) {
    const ri=parseInt(widget.dataset.ri), ci=parseInt(widget.dataset.ci), wi=parseInt(widget.dataset.wi);
    initial[ri].columns[ci].widgets.splice(wi,1);
    render(); return;
  }
});

/* ====== Change handlers ====== */
canvas.addEventListener('change', (e)=>{
  const col    = e.target.closest('.col');
  const widget = e.target.closest('.w');

  if (e.target.classList.contains('wRange')) {
    const ri=parseInt(col.dataset.ri), ci=parseInt(col.dataset.ci);
    initial[ri].columns[ci].width = parseInt(e.target.value);
    col.style.flexBasis = e.target.value+'%';
    serialize(); return;
  }
  if (e.target.classList.contains('spread')) {
    const ri=parseInt(col.dataset.ri), ci=parseInt(col.dataset.ci);
    initial[ri].columns[ci].spread = e.target.checked;
    serialize(); return;
  }
  if (e.target.classList.contains('type')) {
    const ri=parseInt(widget.dataset.ri), ci=parseInt(widget.dataset.ci), wi=parseInt(widget.dataset.wi);
    const type = e.target.value;
    initial[ri].columns[ci].widgets[wi] = {type, cfg:{}};
    injectCfgForm(widget.querySelector('.cfg'), type, {});
    serialize(); return;
  }

  // cfg updates
  if (widget) {
    const ri=parseInt(widget.dataset.ri), ci=parseInt(widget.dataset.ci), wi=parseInt(widget.dataset.wi);
    const type = initial[ri].columns[ci].widgets[wi].type;
    const cfg  = initial[ri].columns[ci].widgets[wi].cfg || {};
    if (type==='logo'){ cfg.image=widget.querySelector('.i-image')?.value||''; cfg.link=widget.querySelector('.i-link')?.value||'/'; }
    else if(type==='search'){ cfg.placeholder=widget.querySelector('.i-ph')?.value||''; }
    else if(type==='lang'){ try{ cfg.items=JSON.parse(widget.querySelector('.i-json').value||'[]'); }catch(e){} }
    else if(type==='social'){ try{ cfg.links=JSON.parse(widget.querySelector('.i-json').value||'[]'); }catch(e){} }
    else if(type==='html'){ cfg.html=widget.querySelector('.i-html')?.value||''; }
    else if(type==='latest'){
      const srcSel = widget.querySelector('.i-source');
      cfg.source = srcSel ? srcSel.value : 'category';
      if (cfg.source === 'post'){
        cfg.post_id = widget.querySelector('.i-postid')?.value||'';
        cfg.cat_slug = '';
        cfg.limit = 1;
      } else {
        cfg.cat_slug = widget.querySelector('.i-cat')?.value||'';
        const lim = parseInt(widget.querySelector('.i-limit')?.value||'4',10);
        cfg.limit = (isFinite(lim)&&lim>0)?lim:4;
        cfg.post_id = '';
      }
      cfg.thumbs = !!widget.querySelector('.i-thumbs')?.checked;
    }
    initial[ri].columns[ci].widgets[wi].cfg = cfg;
    serialize();
  }
});

/* ====== Drag & drop (widgets) ====== */
let dragEl=null;
canvas.addEventListener('dragstart', e=>{
  const w=e.target.closest('.w'); if(!w) return;
  dragEl=w; e.dataTransfer.effectAllowed='move'; w.style.opacity='.5';
});
document.addEventListener('dragend', ()=>{ if(dragEl){dragEl.style.opacity=''; dragEl=null;} });
canvas.addEventListener('dragover', e=>{
  const zone=e.target.closest('.widgets'); if(!zone) return;
  e.preventDefault();
  const after=[...zone.querySelectorAll('.w')].find(b=>{
    const r=b.getBoundingClientRect(); return e.clientX < r.left + r.width/2;
  });
  if(!after) zone.appendChild(dragEl); else zone.insertBefore(dragEl, after);
});
document.addEventListener('drop', ()=>{
  // rebuild model from DOM
  const rows=[...canvas.querySelectorAll('.row')].map(R=>{
    const cols=[...R.querySelectorAll('.col')].map(C=>{
      const width=parseInt(C.querySelector('.wRange').value||100);
      const spread=C.querySelector('.spread').checked;
      const widgets=[...C.querySelectorAll('.w')].map(W=>{
        const type=W.querySelector('.type').value;
        let cfg={};
        if(type==='logo'){ cfg.image=W.querySelector('.i-image')?.value||''; cfg.link=W.querySelector('.i-link')?.value||'/'; }
        else if(type==='search'){ cfg.placeholder=W.querySelector('.i-ph')?.value||''; }
        else if(type==='lang'){ try{ cfg.items=JSON.parse(W.querySelector('.i-json').value||'[]'); }catch(e){} }
        else if(type==='social'){ try{ cfg.links=JSON.parse(W.querySelector('.i-json').value||'[]'); }catch(e){} }
        else if(type==='html'){ cfg.html=W.querySelector('.i-html')?.value||''; }
        else if(type==='latest'){
          cfg.source = W.querySelector('.i-source')?.value || 'category';
          if (cfg.source === 'post'){
            cfg.post_id = W.querySelector('.i-postid')?.value||'';
            cfg.cat_slug = ''; cfg.limit = 1;
          } else {
            cfg.cat_slug = W.querySelector('.i-cat')?.value||'';
            const lim = parseInt(W.querySelector('.i-limit')?.value||'4',10);
            cfg.limit = (isFinite(lim)&&lim>0)?lim:4;
            cfg.post_id = '';
          }
          cfg.thumbs = !!W.querySelector('.i-thumbs')?.checked;
        }
        return {type, cfg};
      });
      return {width, spread, widgets};
    });
    return {columns:cols};
  });
  initial.splice(0, initial.length, ...rows);
  serialize();
});

/* ====== Serialize & preview ====== */
function serialize(){ $id('layout_json').value = JSON.stringify(initial); }
$id('preview').onclick = ()=>{ serialize(); alert($id('layout_json').value); };

render();
</script>
