<?php
// admin/adsenses.php — list of advertisements (adsenses)

require_once __DIR__ . '/../core/auth.php';
require_login();
require_role(['admin']);

require_once __DIR__ . '/../core/db.php';
require_once __DIR__ . '/../core/helpers.php';

/* ---------- tiny helpers ---------- */
function table_exists(string $t): bool {
  return (int) DB::run(
    "SELECT COUNT(*) FROM information_schema.TABLES
     WHERE TABLE_SCHEMA = DATABASE() AND TABLE_NAME = ?", [$t]
  )->fetchColumn() > 0;
}

/* ---------- bootstrap schema (safe to keep) ---------- */
if (!table_exists('ad_clients')) {
  DB::run("
    CREATE TABLE IF NOT EXISTS ad_clients (
      id INT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
      name VARCHAR(120) NOT NULL,
      contact_person VARCHAR(120) NULL,
      phone VARCHAR(60) NULL,
      email VARCHAR(120) NULL,
      website VARCHAR(255) NULL,
      notes TEXT NULL,
      is_active TINYINT(1) NOT NULL DEFAULT 1,
      created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4
  ");
}
if (!table_exists('adsenses')) {
  DB::run("
    CREATE TABLE IF NOT EXISTS adsenses (
      id INT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
      client_id INT UNSIGNED NULL,
      title VARCHAR(255) NOT NULL,
      kind ENUM('image','code') NOT NULL DEFAULT 'image',
      image_path VARCHAR(255) NULL,
      ad_code MEDIUMTEXT NULL,
      width INT UNSIGNED NOT NULL DEFAULT 0,
      height INT UNSIGNED NOT NULL DEFAULT 0,
      is_active TINYINT(1) NOT NULL DEFAULT 1,
      created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
      INDEX idx_client (client_id),
      CONSTRAINT fk_adsense_client
        FOREIGN KEY (client_id) REFERENCES ad_clients(id)
        ON DELETE SET NULL ON UPDATE CASCADE
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4
  ");
}

/* ---------- actions ---------- */
if (isset($_GET['toggle'])) {
  $id = (int)$_GET['toggle'];
  DB::run("UPDATE adsenses SET is_active = 1 - is_active WHERE id=?", [$id]);
  header("Location: adsenses.php");
  exit;
}

/* ---------- data ---------- */
$rows = DB::run("
  SELECT a.id, a.title, a.kind, a.width, a.height, a.is_active, a.created_at,
         COALESCE(c.name,'—') AS client_name
    FROM adsenses a
    LEFT JOIN ad_clients c ON c.id = a.client_id
   ORDER BY a.created_at DESC, a.id DESC
")->fetchAll();

/* ---------- ui ---------- */
include __DIR__ . '/partials/header.php';
include __DIR__ . '/partials/sidebar.php';
?>
<style>
.badge{display:inline-block;padding:4px 10px;border-radius:999px;font-size:12px}
.badge.on{background:#16a34a;color:#fff}
.badge.off{background:#ef4444;color:#fff}
.badge.type{background:#0ea5e9;color:#fff}
.actions a{margin-right:6px}
</style>

<div class="admin-container">
  <h2>All Adsense</h2>

  <div style="margin:12px 0">
    <a class="btn" href="adsense_edit.php">+ New Adsense</a>
  </div>

  <div class="panel">
    <table class="table">
      <thead>
        <tr>
          <th style="width:64px">ID</th>
          <th>Client Info</th>
          <th>Title</th>
          <th style="width:80px">Height</th>
          <th style="width:80px">Width</th>
          <th style="width:90px">Status</th>
          <th style="width:90px">Type</th>
          <th style="width:140px">Created At</th>
          <th style="width:160px">Manage</th>
        </tr>
      </thead>
      <tbody>
        <?php foreach ($rows as $r): ?>
          <tr>
            <td><?= (int)$r['id'] ?></td>
            <td><?= e($r['client_name']) ?></td>
            <td><?= e($r['title']) ?></td>
            <td><?= (int)$r['height'] ?></td>
            <td><?= (int)$r['width'] ?></td>
            <td>
              <?php if ($r['is_active']): ?>
                <span class="badge on">On</span>
              <?php else: ?>
                <span class="badge off">Off</span>
              <?php endif; ?>
            </td>
            <td><span class="badge type"><?= $r['kind']==='image' ? 'Image' : 'Code' ?></span></td>
            <td><?= date('d F Y', strtotime($r['created_at'])) ?></td>
            <td class="actions">
              <a class="btn-sm" href="adsense_edit.php?id=<?= (int)$r['id'] ?>">View</a>
              <a class="btn-sm" href="adsenses.php?toggle=<?= (int)$r['id'] ?>"
                 onclick="return confirm('Toggle status?')">
                <?= $r['is_active'] ? 'Turn Off' : 'Turn On' ?>
              </a>
            </td>
          </tr>
        <?php endforeach; if (!$rows): ?>
          <tr><td colspan="9" style="text-align:center">No data.</td></tr>
        <?php endif; ?>
      </tbody>
    </table>
  </div>
</div>

<?php include __DIR__ . '/partials/footer.php';
